﻿#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.ServiceModel;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

//
// service url:
//
// http://adnmonitor.cloudapp.net/AecInfoService.svc
//
// viewer url:
//
// http://gthelloazure.cloudapp.net/aecmaterialinfo.aspx
//
// aec cloud db demo:
//
// show web service
//
// show cloud database contents
//
// show cmd line access source code
//
// run AecMatInfoAppRevit
//
// show ribbon panel app and drag out of ribbon
//
// run store, show error message
//
// set up comment and mark
//   Wood for Jeremy;Really hard wood;m^3;Switzerland
//
// run store
// update web page
// change mark value, run retrieve
//

namespace AecMatInfoAppRevit
{
  class Util
  {
    const string _url = "http://adnmonitor.cloudapp.net/AecInfoService.svc?wsdl";

    /// <summary>
    /// Retrieve exactly one preselected element, 
    /// or prompt the user to select one if none 
    /// are preselected.
    /// </summary>
    static public Element GetElement( 
      UIDocument uidoc,
      ref string message )
    {
      Element e = null;

      // is exactly one element preselected?

      Selection sel = uidoc.Selection;

      int n = sel.Elements.Size;

      if( 1 < n )
      {
        message = "Please select a single element";
      }
      else if( 1 == n )
      {
        foreach( Element e2 in sel.Elements )
        {
          e = e2;
        }
      }
      else
      {
        // interactive selection:

        try
        {
          Reference r = sel.PickObject(
            ObjectType.Element,
            "Please select an element" );

          e = uidoc.Document.GetElement( 
            r.ElementId );
        }
        catch( OperationCanceledException )
        {
          message = "No element selected";
        }
      }
      return e;
    }

    /// <summary>
    /// Parameter access.
    /// </summary>
    static Parameter GetParameter(
      Element e,
      BuiltInParameter bip )
    {
      Parameter p = e.get_Parameter( bip );

      Debug.Assert( null != p,
        string.Format( "expected valid {0}", bip.ToString() ) );

      return p;
    }

    static string GetStringParameterValue(
      Element e,
      BuiltInParameter bip )
    {
      return GetParameter( e, bip ).AsString();
    }

    static void SetStringParameterValue( 
      Element e, 
      BuiltInParameter bip,
      string s )
    {
      GetParameter( e, bip ).Set( s );
    }

    // sample key: Wood for Jeremy;Really hard wood;m^3;Switzerland

    //const BuiltInParameter _bipCost = BuiltInParameter.ALL_MODEL_COST;
    //const BuiltInParameter _bipKey = BuiltInParameter.ALL_MODEL_TYPE_COMMENTS;

    const BuiltInParameter _bipCost = BuiltInParameter.ALL_MODEL_MARK;
    const BuiltInParameter _bipKey = BuiltInParameter.ALL_MODEL_INSTANCE_COMMENTS;

    static public string GetKey( Element e )
    {
      return GetStringParameterValue( e, _bipKey );
    }

    static public string GetCost( Element e )
    {
      return GetStringParameterValue( e, _bipCost );
    }

    static string[] GetKeys(
      Element e,
      ref string message )
    {
      string key = GetKey( e );

      string[] a = null == key 
        ? null 
        : key.Split( new char[] { ';' } );

      if( null == a || 4 != a.Length )
      {
        message = "Expected element comment property "
          + "to contain a key in the form 'matName;"
          + "matSpec;matUnit;region', for example "
          + "'Wood for Jeremy;Really hard wood;m^3;Switzerland'";

        a = null;
      }

      return a;
    }

    static public void SetCost( Element e, double cost )
    {
      GetParameter( e, _bipCost ).Set( cost.ToString() );
    }

    static AecInfoServiceClient _aecSvrClnt = null; // For AEC Database

    /// <summary>
    /// Initialise cloud service client.
    /// </summary>
    static public void InitialiseServiceClient()
    {
      if( null == _aecSvrClnt )
      {
        System.ServiceModel.Channels.Binding binding 
          = new BasicHttpBinding();

        EndpointAddress a = new EndpointAddress( _url );

        _aecSvrClnt = new AecInfoServiceClient( binding, a );
      }
    }

    /// <summary>
    /// Update AEC material information in the cloud 
    /// for the given element. 
    /// 
    /// The element comment parameter contains the key 
    /// for the cloud database. The element cost parameter 
    /// contains the value to write to it.
    /// </summary>
    static public bool UpdateAecMaterialInfo( 
      Element e,
      ref string message )
    {
      string[] a = GetKeys( e, ref message );

      if( null == a || 4 != a.Length )
      {
        return false;
      }

      float cost;

      try
      {
        cost = float.Parse( Util.GetCost( e ) );
      }
      catch
      {
        message = "Expected element mark property to "
          + "contain a floating-point cost value";

        return false;
      }

      string matName = a[0];
      string matSpec = a[1];
      string matUnit = a[2];
      string region = a[3];

      InitialiseServiceClient();

      _aecSvrClnt.UpdateAecMaterialInfo( matName, matSpec, matUnit, region, cost );

      return true;
    }

    /// <summary>
    /// Retrieve cost per unit from the cloud for the 
    /// given element.
    /// </summary>
    static public bool GetCostPerUnit(
      Element e,
      ref string message )
    {
      string[] a = GetKeys( e, ref message );

      if( null == a || 4 != a.Length )
      {
        return false;
      }

      string matName = a[0];
      string matSpec = a[1];
      string matUnit = a[2];
      string region = a[3];

      InitialiseServiceClient();

      double cost = _aecSvrClnt.GetCostPerUnit( matName, matSpec, matUnit, region );

      using( Transaction tx = new Transaction( e.Document ) )
      {
        tx.Start( "Set AEC element cost" );
        SetCost( e, cost );
        tx.Commit();
      }

      return true;
    }
  }

  [Transaction( TransactionMode.ReadOnly )]
  public class CmdStore : IExternalCommand
  {
    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      Element e = Util.GetElement( uidoc, ref message );

      return null != e 
        && Util.UpdateAecMaterialInfo( e, ref message )
        ? Result.Succeeded
        : Result.Failed;
    }
  }

  [Transaction( TransactionMode.Manual )]
  public class CmdRetrieve : IExternalCommand
  {
    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      Element e = Util.GetElement( uidoc, ref message );

      return null != e
        && Util.GetCostPerUnit( e, ref message )
        ? Result.Succeeded
        : Result.Failed;
    }
  }
}
