#region Namespaces
using System;
using System.Linq;
using System.Collections.Generic;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

namespace BackSectionView
{
  [Transaction( TransactionMode.Automatic )]
  [Regeneration( RegenerationOption.Manual )]
  public class Command : IExternalCommand
  {
    // This function works fine

    private ViewSection GetFrontView( Element oElem )
    {
      // get the bbox of the element

      BoundingBoxXYZ oBBox = oElem.get_BoundingBox( null );

      // get the size of the box

      double oW = oBBox.Max.X - oBBox.Min.X;
      double oD = oBBox.Max.Y - oBBox.Min.Y;
      double oH = oBBox.Max.Z - oBBox.Min.Z;

      // from front with small gap between section and element

      XYZ oMaxPt = new XYZ( -oW / 2.0, oH / 2.0, oD / 2.0 );
      XYZ oMinPt = new XYZ( oW / 2.0, -oH / 2.0, ( -oD / 2.0 ) - 0.2 );

      BoundingBoxXYZ oBBoxView = new BoundingBoxXYZ();
      oBBoxView.Enabled = true;
      oBBoxView.Max = oMaxPt;
      oBBoxView.Min = oMinPt;

      // set the transform

      Transform oTrans = Transform.Identity;

      // find the midpoint of the element

      XYZ oMidPt = 0.5 * ( oBBox.Max + oBBox.Min );

      // set it as origin

      oTrans.Origin = oMidPt;

      // now direction

      XYZ oDirZ = XYZ.BasisY; // new XYZ( 0.0, 1.0, 0.0 );
      XYZ oDirX = -XYZ.BasisX; // new XYZ( -1.0, 0.0, 0.0 );
      XYZ oDirY = XYZ.BasisZ; // new XYZ( 0.0, 0.0, 1.0 );

      oTrans.BasisX = oDirX;
      oTrans.BasisY = oDirY;
      oTrans.BasisZ = oDirZ;

      oBBoxView.Transform = oTrans;

      // create a sectionview and return it

      Document doc = oElem.Document;

      return doc.Create.NewViewSection( oBBoxView );
    }

    // The created section is placed on the front side. The view direction seems okay.

    private ViewSection GetBackView( Element oElem )
    {
      BoundingBoxXYZ oBBox = oElem.get_BoundingBox( null );
      double oW = oBBox.Max.X - oBBox.Min.X;
      double oD = oBBox.Max.Y - oBBox.Min.Y;
      double oH = oBBox.Max.Z - oBBox.Min.Z;
      XYZ oMaxPt = new XYZ( oW / 2.0, oH / 2.0, oD / 2.0 );
      XYZ oMinPt = new XYZ( -oW / 2.0, -oH / 2.0, ( -oD / 2.0 ) - 0.2 );
      BoundingBoxXYZ oBBoxView = new BoundingBoxXYZ();
      oBBoxView.Enabled = true;
      oBBoxView.Max = -oMaxPt;
      oBBoxView.Min = -oMinPt;
      Transform oTrans = Transform.Identity;
      XYZ oMidPt = 0.5 * ( oBBox.Max + oBBox.Min );
      oTrans.Origin = oMidPt;
      XYZ oDirZ = new XYZ( 0.0, 1.0, 0.0 );
      XYZ oDirX = new XYZ( 1.0, 0.0, 0.0 );
      XYZ oDirY = new XYZ( 0.0, 0.0, -1.0 );
      oTrans.BasisX = oDirX;
      oTrans.BasisY = oDirY;
      oTrans.BasisZ = oDirZ;
      oBBoxView.Transform = oTrans;
      Document doc = oElem.Document;
      return doc.Create.NewViewSection( oBBoxView );
    }

    private ViewSection GetTopView( Element elem )
    {
      FamilyInstance famInst = elem as FamilyInstance;

      // For rotating box we must retrieve FacingOrientation
      XYZ fo = famInst.FacingOrientation;

      //BoundingBoxXYZ elemBox = elem.get_BoundingBox(_document.ActiveView);
      BoundingBoxXYZ elemBox = elem.get_BoundingBox( null );

      // get the size of the box

      double boxWidth = elemBox.Max.X - elemBox.Min.X;
      double boxDeep = elemBox.Max.Z - elemBox.Min.Z;
      double boxHeight = elemBox.Max.Y - elemBox.Min.Y;

      XYZ oMaxPt = new XYZ( -boxWidth / 2.0, boxHeight / 2.0, boxDeep / 2.0 );
      XYZ oMinPt = new XYZ( boxWidth / 2.0, -boxHeight / 2.0, -boxDeep / 2.0 );

      // create new BoundigBox

      BoundingBoxXYZ newBoxForView = new BoundingBoxXYZ();
      newBoxForView.Enabled = true;
      newBoxForView.Max = oMinPt;
      newBoxForView.Min = oMaxPt;

      // set the transform
      Transform oTrans = Transform.Identity;

      // find the midpoint of the element

      XYZ midP = 0.5 * ( elemBox.Max + elemBox.Min );

      // origin

      XYZ midWithOffset = new XYZ( midP.X, midP.Y, midP.Z - 0.2 );
      oTrans.Origin = midWithOffset;

      ////direction top - good, only mayby inverted ???

      XYZ oDirX = new XYZ( -fo.Y, fo.X, 0.0 );
      XYZ oDirY = new XYZ( fo.X, fo.Y, 0.0 );
      XYZ oDirZ = new XYZ( 0.0, 0.0, -1.0 );

      //if to use this, and XYZ midWithOffset = new XYZ(midP.X, midP.Y, midP.Z), then some cuts can be seen
      //XYZ oDirX = new XYZ( -fo.Y, fo.X, -2.83540960692694E-15 );
      //XYZ oDirY = new XYZ( fo.X, fo.Y, 0.0 );
      //XYZ oDirZ = new XYZ( -5.89514805537083E-16, 2.77344910411317E-15, -1.0 );

      oTrans.BasisX = oDirX;
      oTrans.BasisY = oDirY;
      oTrans.BasisZ = oDirZ;

      newBoxForView.Transform = oTrans;
      Document doc = elem.Document;
      return doc.Create.NewViewSection( newBoxForView );
    }

    void MakeWallSection( Document doc )
    {
      FilteredElementCollector collector
        = new FilteredElementCollector( doc );

      collector.OfClass( typeof( Wall ) );

      Func<Wall, bool> isLinear
        = w => ( w.Location as LocationCurve ).Curve
          is Line;

      Wall wall = collector
        .OfType<Wall>()
        .First<Wall>( isLinear );

      LocationCurve locC = wall.Location as LocationCurve;
      XYZ startPoint;
      XYZ endPoint;

      // Wall start and end points
      if( !wall.Flipped )
      {
        startPoint = locC.Curve.get_EndPoint( 0 );
        endPoint = locC.Curve.get_EndPoint( 1 );
      }
      else
      {
        endPoint = locC.Curve.get_EndPoint( 0 );
        startPoint = locC.Curve.get_EndPoint( 1 );
      }

      // Normalize to 0 (Z-offsets will be taken care 
      // of later)

      startPoint = startPoint - new XYZ( 0, 0, startPoint.Z );
      endPoint = endPoint - new XYZ( 0, 0, endPoint.Z );

      // Direction is the normal vector to the wall 
      // location curve

      XYZ direction = startPoint - endPoint;
      direction = direction.Normalize();
      direction = direction.CrossProduct( XYZ.BasisZ );

      // Rotation is the angle between direction and 
      // the X-axis

      double rotation = direction.AngleOnPlaneTo(
        XYZ.BasisX, XYZ.BasisZ );

      // Base level and offset

      Level baseLevel = wall.Level;
      double zOffset = baseLevel.Elevation
        + wall.get_Parameter( BuiltInParameter
          .WALL_BASE_OFFSET ).AsDouble();

      // Assumes wall is constrained to layer at top, 
      // if not, different parameters must be checked.

      Level topLevel = doc.get_Element(
        wall.get_Parameter( BuiltInParameter
          .WALL_HEIGHT_TYPE ).AsElementId() ) as Level;

      double height = topLevel.Elevation
        + wall.get_Parameter( BuiltInParameter
          .WALL_TOP_OFFSET ).AsDouble() - zOffset;

      // Create bounding box

      BoundingBoxXYZ bBox = doc.Application.Create
        .NewBoundingBoxXYZ();

      bBox.set_MaxEnabled( 0, true );
      bBox.set_MaxEnabled( 1, true );
      bBox.set_MaxEnabled( 2, true );
      bBox.set_MinEnabled( 0, true );
      bBox.set_MinEnabled( 1, true );
      bBox.set_MinEnabled( 2, true );

      // Transform is as follows: 
      // Z is viewer direction (negative of normal from wall)
      // Y is up.
      // X is determined from these two (note that bBox must be conformal and right handed

      Transform t = Transform.Identity;
      t.BasisZ = direction.Negate();
      t.BasisY = XYZ.BasisZ;
      t.BasisX = XYZ.BasisZ.CrossProduct( direction.Negate() );
      bBox.Transform = t;

      // Transform the boundary points into 
      // coordinate system of the view

      Transform tInv = t.Inverse;
      XYZ boxMin = tInv.OfPoint( startPoint );
      XYZ boxMax = tInv.OfPoint( endPoint );

      // Apply offsets to the boundary points.  
      // Front to back add 6" to half the width of the wall.

      double deltaFrontBack = wall.Width / 2 + 0.5;

      // Add 6" on either side

      double deltaSide = 0.5;

      // Add 6" on top/bottom

      double deltaTopBottom = 0.5;

      // Minimum point subtracts delta, Max point adds them

      bBox.Min = boxMin + new XYZ( -deltaSide,
        zOffset - deltaTopBottom, -deltaFrontBack );

      bBox.Max = boxMax + new XYZ( deltaSide,
        zOffset + height + deltaTopBottom, deltaFrontBack );

      // Make section

      Transaction makeSection = new Transaction( doc,
        "Make section" );

      makeSection.Start();

      ViewSection view = doc.Create.NewViewSection( bBox );

      makeSection.Commit();
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;
      Selection sel = uidoc.Selection;
      Reference r;

      try
      {
        r = sel.PickObject( ObjectType.Element, "Please select an element" );
      }
      catch( OperationCanceledException )
      {
        message = "Nothing selected";
        return Result.Failed;
      }

      ViewSection viewFront = GetFrontView( r.Element );
      ViewSection viewBack = GetBackView( r.Element );

      if( r.Element is FamilyInstance )
      {
        ViewSection viewTop = GetTopView( r.Element );
      }

      return Result.Succeeded;
    }
  }
}
