﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.ApplicationServices;
using Creation = Autodesk.Revit.Creation;

namespace hsbSoft.Revit
{
  [Transaction( TransactionMode.Automatic )]
  [Regeneration( RegenerationOption.Manual )]
  public class BeamMaker : IExternalCommand
  {
    private Application _app;
    private Document _familyDoc;
    private Document _activeDoc;
    private Family _family = null;
    private ExternalCommandData _cmdData;
    private Creation.Application _appCreator;
    private Creation.FamilyItemFactory _familyCreator;
    private string _familyName = "hsbBeam";
    private FamilySymbol _familySymbol = null;

    public Result Execute(
      ExternalCommandData cmdData,
      ref string msg,
      ElementSet elems )
    {
      _cmdData = cmdData;
      Result result = Result.Failed;
      try
      {
        _app = _cmdData.Application.Application;
        _activeDoc = _cmdData.Application.ActiveUIDocument.Document;

        if( CreateFamilyBeam() && _familySymbol != null )
        {
          Curve curve = Line.get_Bound( new XYZ( 0, 0, 0 ), new XYZ( 2, 0, 0 ) );
          _activeDoc.Create.NewFamilyInstance( curve, _familySymbol, GetLevels()[0], Autodesk.Revit.DB.Structure.StructuralType.Beam );
        }
        result = Result.Succeeded;
      }
      catch( System.Exception e )
      {
        msg = e.Message;
        result = Result.Failed;
      }
      return result;
    }
    /// <summary>
    /// Creates a structural framing beam family using the default template
    /// </summary>
    /// <returns></returns>
    private bool CreateFamilyBeam()
    {
      _familyDoc = _activeDoc;
      if( !_familyDoc.IsFamilyDocument )
      {
        string strFamilyTemplate = GetFamilyTemplatePath( "Structural Framing - Beams and Braces.rft" );
        if( string.IsNullOrEmpty( strFamilyTemplate ) )
          return false;
        _familyDoc = _app.NewFamilyDocument( strFamilyTemplate );
        if( null == _familyDoc )
          return false;
      }
      _appCreator = _app.Create;
      _familyCreator = _familyDoc.FamilyCreate;
      Transaction transaction = new Transaction( _familyDoc, _familyName );
      transaction.Start();
      if( CreateVoidExtrusion() == null )
      {
        transaction.RollBack();
        return false;
      }
      //find existing extrusion, and combine them
      FilteredElementCollector collector = new FilteredElementCollector( _familyDoc );
      collector.OfClass( typeof( Extrusion ) );
      IList<Element> lists = collector.ToElements();

      CombinableElementArray elemArray = _app.Create.NewCombinableElementArray();

      foreach( Element elem in lists )
      {
        elemArray.Append( ( CombinableElement ) elem );
      }
      _familyDoc.CombineElements( elemArray );

      if( !LoadFamily() )
      {
        transaction.RollBack();
        return false;
      }
      transaction.Commit();
      return true;
    }

    /// <summary>
    /// Gets the default template path 
    /// </summary>
    private string GetFamilyTemplatePath( 
      string strFamilyTemplateName )
    {
      string strPathForTemplates 
        = _app.FamilyTemplatePath;

      string strTemplateFile 
        = Path.Combine( strPathForTemplates, 
          strFamilyTemplateName );

      if( !File.Exists( strTemplateFile ) )
      {
        strTemplateFile = Path.Combine(
          strPathForTemplates,
          "Metric " + strFamilyTemplateName );
      }

      if( !File.Exists( strTemplateFile ) )
      {
        strTemplateFile = "";
      }

      return strTemplateFile;
    }

    /// <summary>
    /// Gets Family and FamilySymbol from a FamilyDocument
    /// </summary>
    private bool LoadFamily()
    {
      FamilyManager fm = _familyDoc.FamilyManager;
      FamilyType type = fm.NewType( _familyName );
      _family = _familyDoc.LoadFamily( _activeDoc );
      if( _family == null )
        return false;
      _family.Name = _familyName;
      foreach( FamilySymbol fs in _family.Symbols )
      {
        _familySymbol = fs;
        return true;
      }
      return false;
    }
    /// <summary>
    /// Creates a void element in the center of the beam
    /// </summary>
    /// <returns></returns>
    private Extrusion CreateVoidExtrusion()
    {
      CreateExtrusion create = new CreateExtrusion( _app, _familyDoc );
      //curve array for test creates a array of curves in the center
      CurveArrArray curveArrArray = create.CreateCurveArrArray();
      if( curveArrArray == null ) { return null; }
      XYZ normal = new XYZ( 0, 0, 1 );
      XYZ origin = new XYZ( 0, 0, 0 );
      SketchPlane sketchPlane = create.CreateSketchPlane( normal, origin );
      return create.NewExtrusion( false, curveArrArray, sketchPlane, 0, 0.8 );
    }
    /// <summary>
    /// Gets the levels from the active document
    /// </summary>
    /// <returns></returns>
    private IList<Level> GetLevels()
    {
      FilteredElementCollector collector = new FilteredElementCollector( _activeDoc );
      collector.OfCategory( BuiltInCategory.OST_Levels );
      collector.OfClass( typeof( Level ) );
      return collector.ToElements().Cast<Level>().ToList<Level>();
    }
  }
}
