using System;
using System.Collections.Generic;
using Autodesk.Revit;
using Autodesk.Revit.DB;
using Autodesk.Revit.ApplicationServices;

namespace hsbSoft.Revit
{
  /// <summary>
  /// The class is used to create solid extrusion 
  /// </summary>
  public class CreateExtrusion
  {
    #region Class Member Variables
    /// <summary>
    /// store the document
    /// </summary>
    Document m_document;

    /// <summary>
    /// store the application of creation
    /// </summary>
    Autodesk.Revit.Creation.Application m_appCreator;

    /// <summary>
    /// store the FamilyItemFactory of creation
    /// </summary>
    Autodesk.Revit.Creation.FamilyItemFactory m_familyCreator;

    Application m_app = null;
    #endregion

    /// <summary>
    /// The constructor of CreateExtrusion
    /// </summary>
    /// <param name="app">the application</param>
    /// <param name="doc">the document</param>
    public CreateExtrusion( Application app, Document doc )
    {
      m_document = doc;
      m_appCreator = app.Create;
      m_familyCreator = doc.FamilyCreate;
      m_app = app;
    }

    #region Class Implementation
    /// <summary>
    /// The method is used to create a CurveArray with 
    /// four double parameters and one y coordinate value
    /// </summary>
    /// <param name="left">the left value</param>
    /// <param name="right">the right value</param>
    /// <param name="top">the top value</param>
    /// <param name="bottom">the bottom value</param>
    /// <param name="y_coordinate">the y_coordinate value</param>
    /// <returns>CurveArray</returns>
    public CurveArray CreateRectangle(
      double left,
      double right,
      double top,
      double bottom,
      double y_coordinate )
    {
      CurveArray curveArray = m_appCreator.NewCurveArray();
      try
      {
        XYZ p0 = new XYZ( left, y_coordinate, top );
        XYZ p1 = new XYZ( right, y_coordinate, top );
        XYZ p2 = new XYZ( right, y_coordinate, bottom );
        XYZ p3 = new XYZ( left, y_coordinate, bottom );
        Line line1 = m_appCreator.NewLineBound( p0, p1 );
        Line line2 = m_appCreator.NewLineBound( p1, p2 );
        Line line3 = m_appCreator.NewLineBound( p2, p3 );
        Line line4 = m_appCreator.NewLineBound( p3, p0 );
        curveArray.Append( line1 );
        curveArray.Append( line2 );
        curveArray.Append( line3 );
        curveArray.Append( line4 );
        return curveArray;
      }
      catch( Exception e )
      {
        System.Diagnostics.Debug.WriteLine( e.Message );
        return null;
      }
    }

    /// <summary>
    /// The method is used to create a CurveArray along to an origin CurveArray and an offset value
    /// </summary>
    /// <param name="origin">the original CurveArray</param>
    /// <param name="offset">the offset value</param>
    /// <returns>CurveArray</returns>
    public CurveArray CreateCurveArrayByOffset(
      CurveArray origin,
      double offset )
    {
      Line line;
      Line temp;
      int counter = 0;
      CurveArray curveArr = m_appCreator.NewCurveArray();
      XYZ offsetx = new XYZ( offset, 0, 0 );
      XYZ offsetz = new XYZ( 0, 0, offset );
      XYZ p0 = new XYZ();
      XYZ p1 = new XYZ(); ;
      XYZ p2 = new XYZ();
      XYZ p3 = new XYZ();
      foreach( Curve curve in origin )
      {
        temp = curve as Line;
        if( temp != null )
        {
          if( counter == 0 )
          {
            p0 = temp.get_EndPoint( 0 ).Subtract( offsetz ).Subtract( offsetx );
          }
          else if( counter == 1 )
          {
            p1 = temp.get_EndPoint( 0 ).Subtract( offsetz ).Add( offsetx );
          }
          else if( counter == 2 )
          {
            p2 = temp.get_EndPoint( 0 ).Add( offsetx ).Add( offsetz );
          }
          else
          {
            p3 = temp.get_EndPoint( 0 ).Subtract( offsetx ).Add( offsetz );
          }
        }
        counter++;
      }
      line = m_appCreator.NewLineBound( p0, p1 );
      curveArr.Append( line );
      line = m_appCreator.NewLineBound( p1, p2 );
      curveArr.Append( line );
      line = m_appCreator.NewLineBound( p2, p3 );
      curveArr.Append( line );
      line = m_appCreator.NewLineBound( p3, p0 );
      curveArr.Append( line );
      return curveArr;
    }

    /// <summary>
    /// Gets the curve array to create a void extrusion for test
    /// </summary>
    /// <returns></returns>
    public CurveArrArray CreateCurveArrArray()
    {

      List<XYZ> points = new List<XYZ>();
      points.Add( new XYZ( 0, 0, 0 ) );
      points.Add( new XYZ( 0.8, 0, 0 ) );
      points.Add( new XYZ( 0.8, 0.8, 0 ) );
      points.Add( new XYZ( 0, 0.8, 0 ) );
      if( points == null )
        return null;

      CurveArrArray curveArrArray = new CurveArrArray();
      CurveArray curveArray1 = new CurveArray();
      CurveArray curveArray2 = new CurveArray();
      CurveArray curveArray3 = new CurveArray();
      for( int i = 0; i < points.Count; i++ )
      {
        XYZ pt1 = new XYZ( points[i].X, points[i].Y, points[i].Z );
        XYZ pt2;
        if( i == points.Count - 1 )
          pt2 = new XYZ( points[0].X, points[0].Y, points[0].Z );
        else
          pt2 = new XYZ( points[i + 1].X, points[i + 1].Y, points[i + 1].Z );

        Line line = m_app.Create.NewLineBound( pt1, pt2 );
        curveArray1.Append( line );
      }
      // create one rectangular extrusion
      curveArrArray.Append( curveArray1 );
      return curveArrArray;
    }

    /// <summary>
    /// The method is used to create extrusion using FamilyItemFactory.NewExtrusion()
    /// </summary>
    /// <param name="curveArrArray">the CurveArrArray parameter</param>
    /// <param name="workPlane">the reference plane is used to create SketchPlane</param>
    /// <param name="startOffset">the extrusion's StartOffset property</param>
    /// <param name="endOffset">the extrusion's EndOffset property</param>
    /// <returns>the new extrusion</returns>
    public Extrusion NewExtrusion(
      bool isSolid,
      CurveArrArray curveArrArray,
      SketchPlane sketch,
      double startOffset,
      double endOffset )
    {
      Extrusion rectExtrusion = null;
      try
      {
        rectExtrusion = m_familyCreator.NewExtrusion(
          isSolid, curveArrArray, sketch,
          Math.Abs( endOffset - startOffset ) );

        rectExtrusion.StartOffset = startOffset;
        rectExtrusion.EndOffset = endOffset;

        return rectExtrusion;
      }
      catch( Exception ex )
      {
        return null;
      }
    }

    /// <summary>
    /// Create sketch plane for generic model profile
    /// </summary>
    /// <param name="normal">plane normal</param>
    /// <param name="origin">origin point</param>
    /// <returns></returns>
    public SketchPlane CreateSketchPlane( XYZ normal, XYZ origin )
    {
      // First create a Geometry.Plane which need in NewSketchPlane() method
      Plane geometryPlane = m_app.Create.NewPlane( normal, origin );

      if( null == geometryPlane )  // assert the creation is successful
      {
        throw new Exception( "Create the geometry plane failed." );
      }
      
      // Then create a sketch plane using the Geometry.Plane
      SketchPlane plane = m_familyCreator.NewSketchPlane( geometryPlane );
      
      // throw exception if creation failed
      if( null == plane )
      {
        throw new Exception( "Create the sketch plane failed." );
      }
      return plane;
    }
    #endregion
  }
}
