#region Namespaces
using System;
using System.Diagnostics;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
#endregion // Namespaces

namespace BipChecker
{
  /// <summary>
  /// List all accessible parameters on a selected element in a DataGridView.
  /// </summary>
  [Transaction( TransactionMode.ReadOnly )]
  public class BuiltInParamsChecker : IExternalCommand
  {
    const string _type_prompt =
      "This element {0}, so it has both type and"
      + " instance parameters. By default, the instance"
      + " parameters are displayed. If you select 'No',"
      + " the type parameters will be displayed instead."
      + " Would you like to see the instance parameters?";

    /// <summary>
    /// A class used to manage the data of an element parameter.
    /// </summary>
    public class ParameterData
    {
      BuiltInParameter _enum;
      Parameter _parameter;
      string _valueString; // value string or element description in case of an element id

      public ParameterData( BuiltInParameter bip, Parameter parameter, string valueStringOrElementDescription )
      {
        _enum = bip;
        _parameter = parameter;
        _valueString = valueStringOrElementDescription;
      }

      public string Enum
      {
        get { return _enum.ToString(); }
      }

      public string Name
      {
        get { return _parameter.Definition.Name; }
      }

      public string Type
      {
        get
        {
          ParameterType pt = _parameter.Definition.ParameterType; // returns 'Invalid' for 'ElementId'
          string s = ParameterType.Invalid == pt ? "" : "/" + pt.ToString();
          return _parameter.StorageType.ToString() + s;
        }
      }

      public string ReadWrite
      {
        get { return _parameter.IsReadOnly ? "read-only" : "read-write"; }
      }

      public string ValueString
      {
        get { return _valueString; }
      }

      public string Value
      {
        get
        {
          //return _value;
          string s;
          switch( _parameter.StorageType )
          {
            // database value, internal units, e.g. feet:
            case StorageType.Double: s = Util.RealString( _parameter.AsDouble() ); break;
            case StorageType.Integer: s = _parameter.AsInteger().ToString(); break;
            case StorageType.String: s = _parameter.AsString(); break;
            case StorageType.ElementId: s = _parameter.AsElementId().IntegerValue.ToString(); break;
            case StorageType.None: s = "None"; break;
            default: Debug.Assert( false, "unexpected storage type" ); s = string.Empty; break;
          }
          return s;
        }
      }
		}

    /// <summary>
    /// Revit external command to list all valid 
    /// built-in parameters for a given selected 
    /// element.
    /// </summary>
    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Document doc = uidoc.Document;

      // Select element:

      Element e 
        = Util.GetSingleSelectedElementOrPrompt( 
          uidoc );

      if( null == e )
      {
        return Result.Cancelled;
      }
      bool isSymbol = false;
      string family_name = string.Empty;

      // For a family instance, ask user whether to 
      // display instance or type parameters; in a 
      // similar manner, we could add dedicated 
      // switches for Wall --> WallType, 
      // Floor --> FloorType etc. ...

      if( e is FamilyInstance )
      {
        FamilyInstance inst = e as FamilyInstance;
        if( null != inst.Symbol )
        {
          string symbol_name = Util.ElementDescription( 
            inst.Symbol, true );

          family_name = Util.ElementDescription( 
            inst.Symbol.Family, true );

          string msg = string.Format( _type_prompt,
            "is a family instance" );

          if( !Util.QuestionMsg( msg ) )
          {
            e = inst.Symbol;
            isSymbol = true;
          }
        }
      }
      else if( e.CanHaveTypeAssigned() )
      {
        ElementId typeId = e.GetTypeId();
        if( null == typeId )
        {
          Util.InfoMsg( "Element can have a type,"
            + " but the current type is null." );
        }
        else if( ElementId.InvalidElementId == typeId )
        {
          Util.InfoMsg( "Element can have a type,"
            + " but the current type id is the"
            + " invalid element id." );
        }
        else

        {
          Element type = doc.get_Element( typeId );

          if( null == type )
          {
            Util.InfoMsg( "Element has a type,"
              + " but it cannot be accessed." );
          }
          else
          {
            string msg = string.Format( _type_prompt,
              "has an element type" );

            if( !Util.QuestionMsg( msg ) )
            {
              e = type;
              isSymbol = true;
            }
          }
        }
      }

      // Retrieve parameter data:

      SortableBindingList<ParameterData> data 
        = new SortableBindingList<ParameterData>();
      
      {
        WaitCursor waitCursor = new WaitCursor();

        Array bips = Enum.GetValues( 
          typeof( BuiltInParameter ) );

        int n = bips.Length;
        Parameter p;

        foreach( BuiltInParameter a in bips )
        {
          try
          {
            p = e.get_Parameter( a );

    #region Check for external definition
    #if CHECK_FOR_EXTERNAL_DEFINITION
            Definition d = p.Definition;
            ExternalDefinition e = d as ExternalDefinition; // this is never possible
            string guid = ( null == e ) ? null : e.GUID.ToString();
    #endif // CHECK_FOR_EXTERNAL_DEFINITION
            #endregion // Check for external definition

            if( null != p )
            {
              string valueString = 
                (StorageType.ElementId == p.StorageType)
                  ? Util.GetParameterValue2( p, doc )
                  : p.AsValueString();

              data.Add( new ParameterData( a, p, 
                valueString ) );
            }
          }
          catch( Exception ex )
          {
            Debug.Print( 
              "Exception retrieving built-in parameter {0}: {1}",
              a, ex );
          }
        }
      }

      // Display form:

      string description 
        = Util.ElementDescription( e, true ) 
        + ( isSymbol 
          ? " Type" 
          : " Instance" );

      using( BuiltInParamsCheckerForm form 
        = new BuiltInParamsCheckerForm( 
          description, data ) )
      {
        form.ShowDialog();
      }
      return Result.Succeeded;
    }
  }
}
