﻿#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Plumbing;
using Autodesk.Revit.DB.Structure;
using Autodesk.Revit.UI;
#endregion

namespace HellowCS
{
  [Autodesk.Revit.Attributes.Transaction( Autodesk.Revit.Attributes.TransactionMode.Manual )]
  [Autodesk.Revit.Attributes.Regeneration( Autodesk.Revit.Attributes.RegenerationOption.Manual )]

  public class HelloCs : IExternalCommand
  {
    #region Mick sample code
    void MickSampleCode( Pipe pipe, FamilyInstance fi, XYZ dir, XYZ start )
    {
      Document doc = fi.Document;

      // rotate the cap if necessary

      // rotate about Z first

      XYZ pipeHorizontalDirection 
        = new XYZ( dir.X, dir.Y, 0.0 ).Normalize();

      ConnectorSet c2 
        = fi.MEPModel.ConnectorManager.Connectors;

      Connector capConnector = null;

      foreach( Connector c in c2 )
      { 
        capConnector = c; 
      }

      XYZ connectorDirection 
        = -capConnector.CoordinateSystem.BasisZ;

      double zRotationAngle 
        = pipeHorizontalDirection.AngleTo( 
          connectorDirection );

      Transform trf = Transform.get_Rotation( 
        start, XYZ.BasisZ, zRotationAngle );

      XYZ testRotation = trf.OfVector( 
        connectorDirection ).Normalize();

      if( Math.Abs( testRotation.DotProduct( 
        pipeHorizontalDirection ) - 1 ) > 0.00001 )
      {
        zRotationAngle = -zRotationAngle;
      }

      Line axis = doc.Application.Create.NewLineBound( 
        start, start + XYZ.BasisZ );

      ElementTransformUtils.RotateElement( 
        doc, fi.Id, axis, zRotationAngle );

      // Need to rotate vertically?

      if( Math.Abs( dir.DotProduct( XYZ.BasisZ ) ) 
        > 0.000001 )
      {
        // if pipe is straight up and down, 
        // kludge it my way else

        if( dir.X == 0 && dir.Y == 0 && dir.Z != 0 )
        {
          XYZ yaxis = new XYZ( 0.0, 1.0, 0.0 );

          double rotationAngle = dir.AngleTo( yaxis );

          if( dir.Z == 1 )
          {
            rotationAngle = -rotationAngle;
          }

          axis = doc.Application.Create.NewLine( 
            start, yaxis, false );

          ElementTransformUtils.RotateElement( 
            doc, fi.Id, axis, rotationAngle );
        }
        else
        {
          #region sloped pipes

          double rotationAngle = dir.AngleTo( 
            pipeHorizontalDirection );

          XYZ normal = pipeHorizontalDirection
            .CrossProduct( XYZ.BasisZ );

          trf = Transform.get_Rotation( 
            start, normal, rotationAngle );

          testRotation = trf.OfVector( dir ).Normalize();

          if( Math.Abs( testRotation.DotProduct( 
            pipeHorizontalDirection ) - 1 ) < 0.00001 )
          {
            rotationAngle = -rotationAngle;
          }

          axis = doc.Application.Create.NewLineBound( 
            start, start + normal );

          ElementTransformUtils.RotateElement( 
            doc, fi.Id, axis, rotationAngle );

          #endregion
        }
      }

      doc.Regenerate();

      // pick connector on cap:

      ConnectorSet connectors 
        = fi.MEPModel.ConnectorManager.Connectors;

      Debug.Assert( 1 == connectors.Size, 
        "expected only one connector on pipe cap element" );

      Connector cap_end = null;

      foreach( Connector c in connectors )
      { 
        cap_end = c; 
      }

      // pick closest connector on pipe:

      connectors = pipe.ConnectorManager.Connectors;

      Connector pipe_end = null;
      double dist = double.MaxValue;
      foreach( Connector c in connectors )
      {
        XYZ p = c.Origin;
        double d = p.DistanceTo( start );
        if( d < dist )
        {
          dist = d;
          pipe_end = c;
        }
      }
      pipe_end.ConnectTo( cap_end );

      doc.Regenerate();

      // After ConnectTo, the cap size is double, so 
      // need to change the nominal radius to half.

      if( fi != null )
      {
        ConnectorSet pipeConnectors 
          = pipe.ConnectorManager.Connectors;

        Connector pipeEnd = null;

        foreach( Connector c in pipeConnectors )
        {
          pipeEnd = c;
        }
        double radius = pipeEnd.Radius;

        Parameter para = fi.get_Parameter( 
          "Nominal Radius" );

        if( para != null && !para.IsReadOnly )
        {
          para.Set( radius );
        }
      }
    }
    #endregion // Mick sample code

    const string _libFolder = @"C:\ProgramData\Autodesk\RME 2012\Libraries\US Imperial\Pipe\Fittings\Generic";
    const string _rfaExtension = ".rfa";
    const string _capFamilyName = "Cap - Generic";
    const string _capSymbolName = "Standard";

    public Result Execute( ExternalCommandData commandData, ref string message, ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      Document doc = uiapp.ActiveUIDocument.Document;

      //verticla
      CreatePipeWithCap( doc, new XYZ( 10, 10, 0 ), new XYZ( 10, 10, 10 ) );


      // horizontal
      //CreatePipeWithCap(doc, new XYZ(10, 0, 10), new XYZ(20, 0, 10));
      //CreatePipeWithCap(doc, new XYZ(10, 0, 8), new XYZ(10, 10, 8));
      //CreatePipeWithCap(doc, new XYZ(30, 0, 6), new XYZ(20, 0, 6));
      //CreatePipeWithCap(doc, new XYZ(40, 10, 4), new XYZ(40, 0, 4));
      //CreatePipeWithCap(doc, new XYZ(5, 5, 5), new XYZ(10, 13, 5));

      //slanted
      //CreatePipeWithCap(doc, new XYZ(0, 0, 0), new XYZ(6, 0, 15));
      //CreatePipeWithCap(doc, new XYZ(7, 7, 20), new XYZ(23, -5, 13));

      return Result.Succeeded;
    }

    private void CreatePipeWithCap( Document doc, XYZ start, XYZ end )
    {
      PipeType pipeType = new FilteredElementCollector( doc )
                           .OfClass( typeof( ElementType ) )
                               .OfCategory( BuiltInCategory.OST_PipeCurves )
                              .FirstElement() as PipeType;



      Transaction t = new Transaction( doc, "Create Pipe Cap" );
      t.Start();

      Pipe pipe = doc.Create.NewPipe( start, end, pipeType );

      doc.Regenerate();

      //need to update the others
      SetParameterIfPossible( pipe, BuiltInParameter.RBS_CURVE_DIAMETER_PARAM, 0.1667 );
      SetParameterIfPossible( pipe, BuiltInParameter.RBS_PIPE_DIAMETER_PARAM, 0.1667 );
      SetParameterIfPossible( pipe, BuiltInParameter.RBS_CONDUIT_DIAMETER_PARAM, 0.1667 );


      t.Commit();


      List<FamilySymbol> symbols = new List<FamilySymbol>(
        new FilteredElementCollector( doc )
          .OfCategory( BuiltInCategory.OST_PipeFitting )
          .OfClass( typeof( FamilySymbol ) )
          .OfType<FamilySymbol>()
          .Where<FamilySymbol>( s => s.Family.Name.Equals( _capFamilyName ) ) );

      FamilySymbol capSymbol = null;

      if( 0 < symbols.Count )
      {
        capSymbol = symbols[0];
      }
      else
      {
        string filename = Path.Combine( _libFolder, _capFamilyName + _rfaExtension );
        doc.LoadFamilySymbol( filename, _capSymbolName, out capSymbol );
      }
      Debug.Assert( capSymbol.Family.Name.Equals( _capFamilyName ), "expected cap pipe fitting to be of family " + _capFamilyName );

      Transaction t1 = new Transaction( doc, "Cap" );
      t1.Start();

      Autodesk.Revit.DB.XYZ dir = start - end;
      dir = dir.Normalize();

      Options op = new Options();
      op.ComputeReferences = true;
      GeometryObjectArray geoObjs = pipe.get_Geometry( op ).Objects;

      Face bottom = null;

      foreach( GeometryObject geoObj in geoObjs )
      {
        if( geoObj is Solid )
        {
          Solid solid = geoObj as Solid;
          if( null != solid )
          {
            FaceArray faces = solid.Faces;
            foreach( Face face in faces )
            {
              PlanarFace plFace = face as PlanarFace;
              if( null != plFace )
              {
                // get the bottom face
                //if (plFace.Origin.IsAlmostEqualTo(end))
                if( plFace.Project( end ).Distance.Equals( 0.0 ) )
                {
                  bottom = plFace;
                  break;
                }
              }
            }
          }
        }
      }

      XYZ derivatives = bottom.ComputeNormal( new UV( end.X, end.Y ) );
      LocationCurve curve = pipe.Location as LocationCurve;
      Line line = curve.Curve as Line;

      FamilyInstance fi = doc.Create.NewFamilyInstance( start, capSymbol, StructuralType.NonStructural );
      //FamilyInstance fi = doc.Create.NewFamilyInstance(start, capSymbol, dir, pipe, StructuralType.NonStructural);     //this works
      //FamilyInstance fi = doc.Create.NewFamilyInstance(bottom, end, derivatives, capSymbol);     //test version
      //FamilyInstance fi = doc.Create.NewFamilyInstance(bottom, line, capSymbol);     //test version
      //FamilyInstance fi = doc.Create.NewFamilyInstance(bottom, start, dir, capSymbol);

      doc.Regenerate();

      //rotate the cap if necessary

      // rotate about Z first
      XYZ pipeHorizontalDirection = new XYZ( dir.X, dir.Y, 0.0 ).Normalize();

      ConnectorSet c2 = fi.MEPModel.ConnectorManager.Connectors;
      Connector capConnector = null;

      foreach( Connector c in c2 )
      { capConnector = c; }

      XYZ connectorDirection = -capConnector.CoordinateSystem.BasisZ;
      double zRotationAngle = pipeHorizontalDirection.AngleTo( connectorDirection );
      Transform trf = Transform.get_Rotation( start, XYZ.BasisZ, zRotationAngle );
      XYZ testRotation = trf.OfVector( connectorDirection ).Normalize();
      if( Math.Abs( testRotation.DotProduct( pipeHorizontalDirection ) - 1 ) > 0.00001 )
      {
        zRotationAngle = -zRotationAngle;
      }
      Line axis = doc.Application.Create.NewLineBound( start, start + XYZ.BasisZ );
      ElementTransformUtils.RotateElement( doc, fi.Id, axis, zRotationAngle );

      // Need to rotate vertically?
      if( Math.Abs( dir.DotProduct( XYZ.BasisZ ) ) > 0.000001 )
      {
        // now rotate vertically

        double rotationAngle = dir.AngleTo( pipeHorizontalDirection );
        XYZ normal = pipeHorizontalDirection.CrossProduct( XYZ.BasisZ );
        trf = Transform.get_Rotation( start, normal, rotationAngle );
        testRotation = trf.OfVector( dir ).Normalize();
        if( Math.Abs( testRotation.DotProduct( pipeHorizontalDirection ) - 1 ) < 0.00001 )
        {
          rotationAngle = -rotationAngle;
        }

        axis = doc.Application.Create.NewLineBound( start, start + normal );
        ElementTransformUtils.RotateElement( doc, fi.Id, axis, rotationAngle );
      }

      doc.Regenerate();

      // pick connector on cap:
      ConnectorSet connectors = fi.MEPModel.ConnectorManager.Connectors;
      Debug.Assert( 1 == connectors.Size, "expected only one connector on pipe cap element" );

      Connector cap_end = null;

      foreach( Connector c in connectors )
      { cap_end = c; }

      // pick closest connector on pipe:

      connectors = pipe.ConnectorManager.Connectors;

      Connector pipe_end = null;
      double dist = double.MaxValue;
      foreach( Connector c in connectors )
      {
        XYZ p = c.Origin;
        double d = p.DistanceTo( start );
        if( d < dist )
        {
          dist = d;
          pipe_end = c;
        }
      }
      pipe_end.ConnectTo( cap_end );

      doc.Regenerate();


      if( fi != null )  //After ConnectTo(..) Cap size is double, so need to change the norminal radius to half.
      {
        /*Element e = fi as Element;
        if (e != null)
        {
            Parameter para = e.get_Parameter("Nominal Radius");
            if (para != null && !para.IsReadOnly)
            {
                StorageType pType = para.StorageType;
                if (StorageType.Double == pType)
                {
                    double rad = para.AsDouble();     //After ConnectTo(..) Cap size is double
                    para.Set(rad * 0.5);                //so need to change the norminal radius to half.
                }
            }
        }*/
        ConnectorSet pipeConnectors = pipe.ConnectorManager.Connectors;
        Connector pipeEnd = null;
        foreach( Connector c in pipeConnectors )
        {
          pipeEnd = c;
        }
        double radius = pipeEnd.Radius;

        Parameter para = fi.get_Parameter( "Nominal Radius" );
        if( para != null && !para.IsReadOnly )
        {
          para.Set( radius );
        }
      }


      t1.Commit();
    }

    private void SetParameterIfPossible( Element element, BuiltInParameter paramType, double value )
    {
      Parameter p = element.get_Parameter( BuiltInParameter.RBS_CURVE_DIAMETER_PARAM );
      if( p != null )
        p.Set( value );
      /*else
          TaskDialog.Show("Failed to set parameter", "Parameter: " + paramType.ToString());*/
    }
  }
}
