﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Architecture;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;

/* ======================================================================= 
* 【获取最外层的所有墙】 
*  
* 作者：wzhoufeng 1039580989@163.com 
* 时间：2018/5/18 9:03:57 
* 文件名：CmdGetOutermosWallByCreateRoom 
* 版本：V1.0.1 
* 
* 修改者： 时间：  
* 修改说明： 
* ======================================================================== 
*/
namespace YJBim.YJAutoDesign.cmd
{
    /// <summary>
    /// 查找最外层的墙
    /// </summary>
    [Transaction(TransactionMode.Manual)]
    [Regeneration(RegenerationOption.Manual)]
    public class CmdGetOutermosWallByCreateRoom : IExternalCommand
    {
        private Document doc;
        private ExternalCommandData commandData = null;

        public Result Execute(ExternalCommandData commandData, ref string message, ElementSet elements)
        {
            this.commandData = commandData;
            this.doc = commandData.Application.ActiveUIDocument.Document;

            List<ElementId> elementIds = GetOutermostWalls(doc);
            commandData.Application.ActiveUIDocument.Selection.SetElementIds(elementIds);
            return Result.Succeeded;
        }

        /// <summary>
        /// 获取当前模型指定视图内的所有最外层的墙体
        /// </summary>
        /// <param name="doc"></param>
        /// <param name="view">视图,默认是当前激活的视图</param>
        /// <returns></returns>
        public static List<ElementId> GetOutermostWalls(Document doc, View view = null)
        {
            double offset = 1000 / 304.8;
            //获取顶点 最小x，最大y ； 最大x，最小y
            List<Wall> wallList = new FilteredElementCollector(doc).OfClass(typeof(Wall)).Cast<Wall>().ToList();
            double maxX = -1D;
            double minX = -1D;
            double maxY = -1D;
            double minY = -1D;
            wallList.ForEach((wall) =>
            {
                Curve curve = (wall.Location as LocationCurve).Curve;
                XYZ xyz1 = curve.GetEndPoint(0);
                XYZ xyz2 = curve.GetEndPoint(1);

                double _minX = Math.Min(xyz1.X, xyz2.X);
                double _maxX = Math.Max(xyz1.X, xyz2.X);
                double _minY = Math.Min(xyz1.Y, xyz2.Y);
                double _maxY = Math.Max(xyz1.Y, xyz2.Y);

                if (curve.IsCyclic)
                {
                    Arc arc = curve as Arc;
                    double _radius = arc.Radius;
                    //粗略对x和y 加/减
                    _maxX += _radius;
                    _minX -= _radius;
                    _maxY += _radius;
                    _minY += _radius;
                }

                if (minX == -1) minX = _minX;
                if (maxX == -1) maxX = _maxX;
                if (maxY == -1) maxY = _maxY;
                if (minY == -1) minY = _minY;

                if (_minX < minX) minX = _minX;
                if (_maxX > maxX) maxX = _maxX;
                if (_maxY > maxY) maxY = _maxY;
                if (_minY < minY) minY = _minY;
            });
            minX -= offset;
            maxX += offset;
            minY -= offset;
            maxY += offset;

            CurveArray curves = new CurveArray();
            Line line1 = Line.CreateBound(new XYZ(minX, maxY, 0), new XYZ(maxX, maxY, 0));
            Line line2 = Line.CreateBound(new XYZ(maxX, maxY, 0), new XYZ(maxX, minY, 0));
            Line line3 = Line.CreateBound(new XYZ(maxX, minY, 0), new XYZ(minX, minY, 0));
            Line line4 = Line.CreateBound(new XYZ(minX, minY, 0), new XYZ(minX, maxY, 0));
            curves.Append(line1); curves.Append(line2); curves.Append(line3); curves.Append(line4);

            using (TransactionGroup group = new TransactionGroup(doc))
            {

                Room newRoom = null;
                RoomTag tag1 = null;

                group.Start("find outermost walls");
                using (Transaction transaction = new Transaction(doc, "createNewRoomBoundaryLines"))
                {
                    transaction.Start();
                    if (view == null)
                        view = doc.ActiveView;
                    SketchPlane sketchPlane = SketchPlane.Create(doc, view.GenLevel.Id);

                    ModelCurveArray modelCaRoomBoundaryLines = doc.Create.NewRoomBoundaryLines(sketchPlane, curves, view);


                    //创建房间的坐标点
                    XYZ point = new XYZ(minX + 600 / 304.8, maxY - 600 / 304.8, 0);

                    //根据选中点，创建房间   当前视图的楼层doc.ActiveView.GenLevel
                    newRoom = doc.Create.NewRoom(view.GenLevel, new UV(point.X, point.Y));

                    if (newRoom == null)
                    {
                        string msg = "创建房间失败。";
                        TaskDialog.Show("xx", msg);
                        transaction.RollBack();
                        return null;
                    }
                    tag1 = doc.Create.NewRoomTag(new LinkElementId(newRoom.Id), new UV(point.X, point.Y), view.Id);
                    transaction.Commit();
                }
                //获取房间的墙体
                List<ElementId> elementIds = DetermineAdjacentElementLengthsAndWallAreas(doc, newRoom);
                group.RollBack();//撤销
                return elementIds;
            }

        }
        /// <summary>
        /// 过滤出需要的墙体
        /// </summary>
        /// <param name="doc"></param>
        /// <param name="room"></param>
        /// <returns></returns>
        static List<ElementId> DetermineAdjacentElementLengthsAndWallAreas(Document doc, Room room)
        {
            List<ElementId> elementIds = new List<ElementId>();

            IList<IList<BoundarySegment>> boundaries
              = room.GetBoundarySegments(new SpatialElementBoundaryOptions());

            int n = boundaries.Count;//.Size;

            int iBoundary = 0, iSegment;

            foreach (IList<BoundarySegment> b in boundaries)
            {
                ++iBoundary;
                iSegment = 0;
                foreach (BoundarySegment s in b)
                {
                    ++iSegment;
                    Element neighbour = doc.GetElement(s.ElementId);// s.Element;
                    Curve curve = s.GetCurve();//.Curve;
                    double length = curve.Length;

                    if (neighbour is Wall)
                    {
                        Wall wall = neighbour as Wall;

                        Parameter p = wall.get_Parameter(
                          BuiltInParameter.HOST_AREA_COMPUTED);

                        double area = p.AsDouble();

                        LocationCurve lc
                          = wall.Location as LocationCurve;

                        double wallLength = lc.Curve.Length;

                        elementIds.Add(wall.Id);
                    }
                }
            }
            return elementIds;
        }

    }





}
