﻿#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Plumbing;
using Autodesk.Revit.DB.Structure;
using Autodesk.Revit.UI;
#endregion

namespace CreatePipeCap
{
  [Transaction( TransactionMode.Manual )]
  [Regeneration( RegenerationOption.Manual )]
  public class Command : IExternalCommand
  {
    const string _libFolder = "C:/Documents and Settings"
      + "/All Users/Application Data/Autodesk/RME 2011"
      + "/Metric Library/Pipe/Fittings/Generic";

    const string _rfaExtension = ".rfa";
    const string _capFamilyName = "M_Cap - Generic";
    const string _capSymbolName = "Standard";

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      Document doc = uiapp.ActiveUIDocument.Document;

      PipeType pipeType
        = new FilteredElementCollector( doc )
        .OfClass( typeof( ElementType ) )
        .OfCategory( BuiltInCategory.OST_PipeCurves )
        .FirstElement() as PipeType;

      XYZ start = new XYZ( 0, 0, 0 );
      XYZ end = new XYZ( 6, 0, 4 );

      Transaction t = new Transaction( doc, 
        "Create Pipe Cap" );

      t.Start();

      Pipe pipe = doc.Create.NewPipe( 
        start, end, pipeType );

      pipe.get_Parameter( 
        BuiltInParameter.RBS_PIPE_DIAMETER_PARAM )
        .Set( 0.1667 ); // revise to 2" dia pipe

      // get cap symbol:

      // get cap symbol family:
      //IEnumerable<Family> families =
      //  (new FilteredElementCollector( doc )
      //  .OfClass( typeof( Family ) ))
      //  .OfType<Family>()
      //  .Where<Family>( f => f.Name.Equals( _capFamilyName ) );
      //FamilySymbolFilter fsf
      //  = new FamilySymbolFilter(
      //      families.First<Family>().Id );

      List<FamilySymbol> symbols = new List<FamilySymbol>(
        new FilteredElementCollector( doc )
          .OfCategory( BuiltInCategory.OST_PipeFitting )
          .OfClass( typeof( FamilySymbol ) )
          .OfType<FamilySymbol>()
          .Where<FamilySymbol>( s 
            => s.Family.Name.Equals( _capFamilyName ) ) );

      FamilySymbol capSymbol = null;

      if( 0 < symbols.Count )
      {
        capSymbol = symbols[0];
      }
      else 
      {
        string filename = Path.Combine( _libFolder, 
          _capFamilyName + _rfaExtension );

        // requires a transaction, obviously:

        doc.LoadFamilySymbol( filename, _capSymbolName, 
          out capSymbol );
      }

      Debug.Assert( capSymbol.Family.Name.Equals( _capFamilyName ),
        "expected cap pipe fitting to be of family " + _capFamilyName );

      FamilyInstance fi = doc.Create.NewFamilyInstance( 
        end, capSymbol, StructuralType.NonStructural );

      // pick connector on cap:

      ConnectorSet connectors 
        = fi.MEPModel.ConnectorManager.Connectors;

      Debug.Assert( 1 == connectors.Size,
        "expected nly one connector on pipe cap element" );

      Connector cap_end = null;

      foreach( Connector c in connectors )
      {
        cap_end = c;
      }

      // pick closest connector on pipe:

      connectors = pipe.ConnectorManager.Connectors;

      Connector pipe_end = null;

      // the order of connectors returned by the 
      // connector manager may change, so we 
      // always use a location (or even more 
      // information if several connectors are 
      // at the same location) to get the right
      // connector!

      double dist = double.MaxValue;

      foreach( Connector c in connectors )
      {
        XYZ p = c.Origin;
        double d = p.DistanceTo( end );

        if( d < dist )
        {
          dist = d;
          pipe_end = c;
        }
        break;
      }

      cap_end.ConnectTo( pipe_end );

      t.Commit();

      return Result.Succeeded;
    }
  }
}
