﻿#region Namespaces
using System;
using System.Diagnostics;
using System.Collections.Generic;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Mechanical;
using Autodesk.Revit.DB.Plumbing;
using Autodesk.Revit.UI;
#endregion

namespace CreateTakeoff
{
  /// <summary>
  /// Revit external command to demonstrate 
  /// NewTakeOffFitting method.
  /// 
  /// Create two pipes in a new document
  /// and call NewTakeOffFitting to add
  /// a takeoff to connect them.
  /// </summary>
  [Transaction( TransactionMode.Manual )]
  [Regeneration( RegenerationOption.Manual )]
  public class Command : IExternalCommand
  {
    #region Message handling
    const string _caption = "Create Takeoff";

    /// <summary>
    /// MessageBox or Revit TaskDialog 
    /// wrapper for error message.
    /// </summary>
    public static void ErrorMsg( string msg )
    {
      Debug.WriteLine( msg );

      TaskDialog d = new TaskDialog( _caption );
      d.MainIcon = TaskDialogIcon.TaskDialogIconWarning;
      d.MainInstruction = msg;
      d.Show();
    }
    #endregion // Message handling

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      Result result = Result.Failed;

      UIApplication uiapp = commandData.Application;
      Document doc = uiapp.ActiveUIDocument.Document;

      Transaction t = null;

      Autodesk.Revit.Creation.Document creDoc
        = doc.Create;

      try
      {
        // determine duct type to use:

        FilteredElementCollector collector
          = new FilteredElementCollector( doc );

        collector.OfCategory(
          BuiltInCategory.OST_DuctCurves );

        collector.OfClass( typeof(
          ElementType ) );

        DuctType ductType
          = collector.FirstElement() as DuctType;

        if( null == ductType )
        {
          ErrorMsg( "No duct types found." );
        }
        else
        {
          t = new Transaction( doc, _caption );

          t.Start();

          // create duct1 along X axis 
          // from (0,0,0) to (6,0,0):

          XYZ start = XYZ.Zero;
          XYZ end = start + 6 * XYZ.BasisX;
          XYZ mid = 0.5 * ( start + end );

          Duct duct1 = creDoc.NewDuct(
            start, end, ductType );

          // create duct2 parallel to Y 
          // axis from (1,2,0) to (1,4,0):

          start = mid + 2 * XYZ.BasisY;
          end = start + 2 * XYZ.BasisY;

          Duct duct2 = creDoc.NewDuct(
            start, end, ductType );

          // pick closest connector on duct2:

          Connector duct2_start = null;

          // just picking the first one is unreliable!
          // the order of connector returned by the 
          // connector manager may change!
          // always use a location (or even more 
          // information if 2 connectors are at 
          // the same location) to get the right
          // connector!

          double dist = double.MaxValue;

          foreach( Connector c in
            duct2.ConnectorManager.Connectors )
          {
            XYZ p = c.Origin;
            double d = p.DistanceTo( mid );

            if( d < dist )
            {
              dist = d;
              duct2_start = c;
            }
            break;
          }

          // create takeoff from duct1 to duct2:

          FamilyInstance takeoff
            = creDoc.NewTakeoffFitting(
              duct2_start, duct1 );

          t.Commit();

          result = Result.Succeeded;
        }
      }
      catch( Exception ex )
      {
        if( null != t )
        {
          t.RollBack();
        }
        message = ex.Message;
      }
      return result;
    }
  }
}
