﻿#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Mechanical;
using Autodesk.Revit.DB.Plumbing;
using Autodesk.Revit.DB.Structure;
using Autodesk.Revit.UI;
#endregion

namespace CreateTakeoff
{
  public class Message
  {
    public const string Caption = "Create Takeoff";

    /// <summary>
    /// Revit TaskDialog wrapper for error message.
    /// </summary>
    public static void ShowError( string msg )
    {
      Debug.WriteLine( msg );

      TaskDialog d = new TaskDialog( Caption );
      d.MainIcon = TaskDialogIcon.TaskDialogIconWarning;
      d.MainInstruction = msg;
      d.Show();
    }
  }

  #region CommandDuct
#if CommandDuct
  /// <summary>
  /// Revit external command to demonstrate 
  /// NewTakeOffFitting method for ducts.
  /// 
  /// Create two ducts in a new document
  /// and call NewTakeOffFitting to add
  /// a takeoff to connect them.
  /// </summary>
  [Transaction( TransactionMode.Manual )]
  [Regeneration( RegenerationOption.Manual )]
  public class CommandDuct : IExternalCommand
  {
    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      Result result = Result.Failed;

      UIApplication uiapp = commandData.Application;
      Document doc = uiapp.ActiveUIDocument.Document;

      Transaction t = null;

      Autodesk.Revit.Creation.Document creDoc
        = doc.Create;

      try
      {
        // determine duct type to use:

        FilteredElementCollector collector
          = new FilteredElementCollector( doc );

        collector.OfCategory(
          BuiltInCategory.OST_DuctCurves );

        collector.OfClass( typeof(
          ElementType ) );

        DuctType ductType
          = collector.FirstElement() as DuctType;

        if( null == ductType )
        {
          Message.ShowError( "No duct types found." );
        }
        else
        {
          t = new Transaction( doc, Message.Caption );

          t.Start();

          // create duct1 along X axis 
          // from (0,0,0) to (6,0,0):

          XYZ start = XYZ.Zero;
          XYZ end = start + 6 * XYZ.BasisX;
          XYZ mid = 0.5 * ( start + end );

          Duct duct1 = creDoc.NewDuct(
            start, end, ductType );

          // create duct2 parallel to Y 
          // axis from (1,2,0) to (1,4,0):

          start = mid + 2 * XYZ.BasisY;
          end = start + 2 * XYZ.BasisY;

          Duct duct2 = creDoc.NewDuct(
            start, end, ductType );

          // pick closest connector on duct2:

          Connector duct2_start = null;

          // just picking the first one is unreliable!
          // the order of connector returned by the 
          // connector manager may change!
          // always use a location (or even more 
          // information if 2 connectors are at 
          // the same location) to get the right
          // connector!

          double dist = double.MaxValue;

          foreach( Connector c in
            duct2.ConnectorManager.Connectors )
          {
            XYZ p = c.Origin;
            double d = p.DistanceTo( mid );

            if( d < dist )
            {
              dist = d;
              duct2_start = c;
            }
            break;
          }

          // create takeoff from duct1 to duct2:

          FamilyInstance takeoff
            = creDoc.NewTakeoffFitting(
              duct2_start, duct1 );

          t.Commit();

          result = Result.Succeeded;
        }
      }
      catch( Exception ex )
      {
        if( null != t )
        {
          t.RollBack();
        }
        message = ex.Message;
      }
      return result;
    }
  }
#endif // CommandDuct
  #endregion // CommandDuct

  #region CommandInsertPipeSpudInstance
#if CommandInsertPipeSpudInstance
  [Transaction( TransactionMode.Manual )]
  [Regeneration( RegenerationOption.Manual )]
  public class CommandInsertPipeSpudInstance : IExternalCommand
  {
    string _spudFamilyName = "Pipe Spud";
    string _spudFamilyPath = "C:/a/j/adn/case/bsd/1266201/attach/";
    string _rfaExt = ".rfa";
    string _spudSymbolName = "Standard";

    const BuiltInCategory _bicPipeCurves = BuiltInCategory.OST_PipeCurves;
    const BuiltInCategory _bicPipeFitting = BuiltInCategory.OST_PipeFitting;

    const BuiltInParameter _bipDiam = BuiltInParameter.RBS_PIPE_DIAMETER_PARAM;

    private XYZ FindDirection1( XYZ start, XYZ end )
    {
      double xx = end.X - start.X;
      double yy = end.Y - start.Y;
      double zz = end.Z - start.Z;
      double temp = Math.Sqrt( xx * xx + yy * yy + zz * zz );
      XYZ dir = new XYZ( xx / temp, yy / temp, zz / temp );
      return dir;
    }

    private XYZ FindDirection( XYZ start, XYZ end )
    {
      XYZ v = end - start;
      return v.Normalize();
    }

    public Result Execute( 
      ExternalCommandData commandData, 
      ref string message, 
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      Document doc = uiapp.ActiveUIDocument.Document;
      
      Autodesk.Revit.Creation.Document creDoc = doc.Create;

      Transaction transaction 
        = new Transaction( doc, "CreatePipe" );

      transaction.Start();

      // get pipe type:

      FilteredElementCollector collector 
        = new FilteredElementCollector( doc )
          .OfCategory( _bicPipeCurves )
          .OfClass( typeof( ElementType ) );

      PipeType pipeType = collector.FirstElement() as PipeType;

      // create pipes:

      XYZ start = new XYZ( 0, 0, 0 );
      XYZ end = new XYZ( 6, 0, 4 );
      XYZ mid = new XYZ( 3, 3, 2 );
      XYZ olet = new XYZ( 3, 6, 2 );

      Pipe pipe1 = doc.Create.NewPipe( start, end, pipeType );

      if( pipe1 != null )
      {
        // 6" dia pipe
        pipe1.get_Parameter( _bipDiam ).Set( 0.5 ); 
      }

      Pipe outlet = doc.Create.NewPipe( mid, olet, pipeType );

      if( outlet != null )
      {
        // 2" dia pipe
        outlet.get_Parameter( _bipDiam ).Set( 0.1666 ); 
      }

      XYZ dir = FindDirection( mid, olet );

      // get spud symbol:

      List<FamilySymbol> symbols = new List<FamilySymbol>( 
        new FilteredElementCollector( doc )
          .OfCategory( _bicPipeFitting )
          .OfClass( typeof( FamilySymbol ) )
          .OfType<FamilySymbol>()
          .Where<FamilySymbol>( s => s.Family.Name.Equals( _spudFamilyName ) ) );

      FamilySymbol spudSymbol = null;

      if( 0 < symbols.Count )
      {
        spudSymbol = symbols[0];
      }
      else 
      {
        string filename = Path.Combine( _spudFamilyPath, 
          _spudFamilyName + _rfaExt );

        // requires a transaction, obviously:

        doc.LoadFamilySymbol( filename, _spudSymbolName, 
          out spudSymbol );
      }

      Debug.Assert( spudSymbol.Family.Name.Equals( _spudFamilyName ),
        "expected pipe fitting to be of family " + _spudFamilyName );

      //FamilyInstance fi = creDoc.NewFamilyInstance( mid, spudSymbol, dir, opipe, StructuralType.NonStructural );

      mid = new XYZ( 3, 0, 2 );

      FamilyInstance fi = creDoc.NewFamilyInstance( mid, spudSymbol, StructuralType.NonStructural );

      transaction.Commit();

      return Result.Succeeded;
    }
  }
#endif // CommandInsertPipeSpudInstance
  #endregion // CommandInsertPipeSpudInstance

  /// <summary>
  /// Revit external command to demonstrate 
  /// NewTakeOffFitting method for pipes.
  /// 
  /// Create two pipes in a new document
  /// and call NewTakeOffFitting to add
  /// a takeoff to connect them.
  /// </summary>
  [Transaction( TransactionMode.Manual )]
  [Regeneration( RegenerationOption.Manual )]
  public class Command : IExternalCommand
  {
    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      Result result = Result.Failed;

      UIApplication uiapp = commandData.Application;
      Document doc = uiapp.ActiveUIDocument.Document;

      Transaction t = null;

      Autodesk.Revit.Creation.Document creDoc
        = doc.Create;

      try
      {
        // determine pipe type to use:

        FilteredElementCollector collector
          = new FilteredElementCollector( doc );

        collector.OfCategory(
          BuiltInCategory.OST_PipeCurves );

        collector.OfClass( typeof(
          ElementType ) );

        PipeType pipeType
          = collector.FirstElement() as PipeType;

        if( null == pipeType )
        {
          Message.ShowError( "No pipe types found." );
        }
        else
        {
          t = new Transaction( doc, Message.Caption );

          t.Start();

          // create pipe1 along X axis 
          // from (0,0,0) to (6,0,0):

          XYZ start = XYZ.Zero;
          XYZ end = start + 6 * XYZ.BasisX;
          XYZ mid = 0.5 * ( start + end );

          Pipe pipe1 = creDoc.NewPipe(
            start, end, pipeType );

          // create pipe2 parallel to Y 
          // axis from (1,2,0) to (1,4,0):

          start = mid + 2 * XYZ.BasisY;
          end = start + 2 * XYZ.BasisY;

          Pipe pipe2 = creDoc.NewPipe(
            start, end, pipeType );

          // pick closest connector on pipe2:

          Connector pipe2_start = null;

          // just picking the first one is unreliable!
          // the order of connector returned by the 
          // connector manager may change!
          // always use a location (or even more 
          // information if 2 connectors are at 
          // the same location) to get the right
          // connector!

          double dist = double.MaxValue;

          foreach( Connector c in
            pipe2.ConnectorManager.Connectors )
          {
            XYZ p = c.Origin;
            double d = p.DistanceTo( mid );

            if( d < dist )
            {
              dist = d;
              pipe2_start = c;
            }
            break;
          }

          // create takeoff from pipe1 to pipe2:

          FamilyInstance takeoff
            = creDoc.NewTakeoffFitting(
              pipe2_start, pipe1 );

          t.Commit();

          result = Result.Succeeded;
        }
      }
      catch( Exception ex )
      {
        if( null != t )
        {
          t.RollBack();
        }
        message = ex.Message;
      }
      return result;
    }
  }
}
