#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

namespace CreateWallSectionView
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    /// <summary>
    /// Return a section box for a view perpendicular 
    /// to the given wall location line.
    /// </summary>
    BoundingBoxXYZ GetSectionViewPerpendiculatToWall(
      Wall wall )
    {
      LocationCurve lc = wall.Location 
        as LocationCurve;

      // Using 0.5 and "true" to specify that the 
      // parameter is normalized places the transform
      // origin at the center of the location curve

      Transform curveTransform = lc.Curve
        .ComputeDerivatives( 0.5, true );

      // The transform contains the location curve
      // mid-point and tangent, and we can obtain
      // its normal in the XY plane:

      XYZ origin = curveTransform.Origin;
      XYZ viewdir = curveTransform.BasisX.Normalize();
      XYZ up = XYZ.BasisZ;
      XYZ right = up.CrossProduct( viewdir );

      // Set up view transform, assuming wall's "up" 
      // is vertical. For a non-vertical situation 
      // such as section through a sloped floor, the 
      // surface normal would be needed

      Transform transform = Transform.Identity;
      transform.Origin = origin;
      transform.BasisX = right;
      transform.BasisY = up;
      transform.BasisZ = viewdir;

      BoundingBoxXYZ sectionBox = new BoundingBoxXYZ();
      sectionBox.Transform = transform;

      // Min & Max X values -10 and 10 define the 
      // section line length on each side of the wall.
      // Max Y (12) is the height of the section box.
      // Max Z (5) is the far clip offset.

      double d = wall.WallType.Width;
      BoundingBoxXYZ bb = wall.get_BoundingBox( null );
      double minZ = bb.Min.Z;
      double maxZ = bb.Max.Z;
      double h = maxZ - minZ;

      sectionBox.Min = new XYZ( -2 * d, -1, 0 );
      sectionBox.Max = new XYZ( 2 * d, h + 1, 5 );

      return sectionBox;
    }

    /// <summary>
    /// Return a section box for a view parallel
    /// to the given wall location line.
    /// </summary>
    BoundingBoxXYZ GetSectionViewParallelToWall(
      Wall wall )
    {
      LocationCurve lc = wall.Location 
        as LocationCurve;
      
      Curve curve = lc.Curve;

      // view direction sectionBox.Transform.BasisZ
      // up direction sectionBox.Transform.BasisY
      // right hand is computed so that (right, up, view direction) form a left handed coordinate system.  
      // crop region projections of BoundingBoxXYZ.Min and BoundingBoxXYZ.Max onto the view cut plane
      // far clip distance difference of the z-coordinates of BoundingBoxXYZ.Min and BoundingBoxXYZ.Max

      XYZ p = curve.get_EndPoint( 0 );
      XYZ q = curve.get_EndPoint( 1 );
      XYZ v = q - p;

      BoundingBoxXYZ bb = wall.get_BoundingBox( null );
      double minZ = bb.Min.Z;
      double maxZ = bb.Max.Z;

      double w = v.GetLength();
      double h = maxZ - minZ;
      double d = wall.WallType.Width;
      double offset = 0.1 * w;

      XYZ min = new XYZ( -w, minZ - offset, -offset );
      //XYZ max = new XYZ( w, maxZ + offset, 0 ); // section view dotted line in center of wall
      XYZ max = new XYZ( w, maxZ + offset, offset ); // section view dotted line offset from center of wall

      XYZ midpoint = p + 0.5 * v;
      XYZ walldir = v.Normalize();
      XYZ up = XYZ.BasisZ;
      XYZ viewdir = walldir.CrossProduct( up );

      Transform t = Transform.Identity;
      t.Origin = midpoint;
      t.BasisX = walldir;
      t.BasisY = up;
      t.BasisZ = viewdir;

      BoundingBoxXYZ sectionBox = new BoundingBoxXYZ();
      sectionBox.Transform = t;
      sectionBox.Min = min;
      sectionBox.Max = max;

      return sectionBox;
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Document doc = uidoc.Document;

      // Retrieve wall from selection set

      Selection sel = uidoc.Selection;

      SelElementSet set = sel.Elements;

      Wall wall = null;

      if( 1 == set.Size )
      {
        foreach( Element e in set )
        {
          wall = e as Wall;
        }
      }

      if( null == wall )
      {
        message = "Please select exactly one wall element.";

        return Result.Failed;
      }

      // Ensure wall is straight

      LocationCurve lc = wall.Location as LocationCurve;

      Line line = lc.Curve as Line;

      if( null == line )
      {
        message = "Unable to retrieve wall location line.";

        return Result.Failed;
      }

      // Determine view family type to use

      ViewFamilyType vft
        = new FilteredElementCollector( doc )
          .OfClass( typeof( ViewFamilyType ) )
          .Cast<ViewFamilyType>()
          .FirstOrDefault<ViewFamilyType>( x =>
            ViewFamily.Section == x.ViewFamily );

      // Determine section box

      BoundingBoxXYZ sectionBox
        = GetSectionViewParallelToWall( wall );
        //= GetSectionViewPerpendiculatToWall( wall );

      // Create wall section view

      using( Transaction tx = new Transaction( doc ) )
      {
        tx.Start( "Create Wall Section View" );

        ViewSection.CreateSection( doc, vft.Id, sectionBox );

        tx.Commit();
      }
      return Result.Succeeded;
    }
  }
}
