#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

namespace CustomExporterXml
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    /// <summary>
    /// The main (and only :-) command 
    /// of our export sample
    /// </summary>
    public Result Execute(
      ExternalCommandData data,
      ref string msg,
      ElementSet elements )
    {
      // This command requires an active document

      UIDocument uidoc = data.Application
        .ActiveUIDocument;

      if( uidoc == null )
      {
        msg = "Please run this command in an active project document.";
        return Result.Failed;
      }

      Document doc = uidoc.Document;

      // Collect all 3D views in the document 
      // (there must be at least one). The collection
      // will be listed in the option dialog for the
      // user to choose the ones to include in the
      // export.

      List<View3D> views = new List<View3D>(
        new FilteredElementCollector( doc )
          .OfClass( typeof( View3D ) )
          .Cast<View3D>()
          .Where<View3D>( v =>
            v.CanBePrinted && !v.IsTemplate ) );

      int n = views.Count;

      if( 0 == n )
      {
        msg = "There are no 3D views in the document!";
        return Result.Failed;
      }

      // Initiate output with default path to 
      // MyDocuments and the current document name

      string defaultName = Path.ChangeExtension(
        doc.Title, ".xml" );

      string defaultFolder = System.Environment
        .GetFolderPath(
          Environment.SpecialFolder.MyDocuments );

      string defaultPath = Path.Combine(
        defaultFolder, defaultName );

      // Show the option dialog to let the user choose 
      // a file, views, and options to export with

      ExportOptionsDialog optionsDlg
        = new ExportOptionsDialog( defaultPath,
          views.Select<View3D, string>( v => v.Name ) );

      if( System.Windows.Forms.DialogResult.Cancel
        == optionsDlg.ShowDialog() )
      {
        return Result.Cancelled;
      }

      // Get the list of views to be exported

      IList<int> selectedViews
        = optionsDlg.GetCheckedViewsIndices();

      n = ( null == selectedViews )
        ? 0
        : selectedViews.Count;

      if( 0 == n )
      {
        msg = "No views selected.";
        return Result.Cancelled;
      }

      List<ElementId> viewsToExport
        = new List<ElementId>( n );

      foreach( int index in selectedViews )
      {
        viewsToExport.Add( views[index].Id );
      }

      // Instantiate our custom context

      ExportContext context = new ExportContext(
        doc, optionsDlg.GetFilePath(),
        optionsDlg.GetOptions() );

      // Instantiate a custom exporter with output 
      // context given as the argument

      using( CustomExporter exporter
        = new CustomExporter( doc, context ) )
      {
        // Tell the exportrd whether we need face info.
        // If not, it is better to exclude them, since 
        // processing faces takes significant time and 
        // memory. In any case, tessellated polymeshes
        // can be exported (and will be sent to the 
        // context).

        exporter.IncludeFaces = optionsDlg.GetOptions()
          .IncludeFaces;

        try
        {
          exporter.Export( viewsToExport );
        }
        catch( System.Exception ex )
        {
          msg = "Exception: " + ex.Message;
          return Result.Failed;
        }
      }
      return Result.Succeeded;
    }
  }
}
