﻿using System;
using System.IO;
using System.Text;
using System.Xml;
using System.Collections.Generic;
using System.Diagnostics;

using Autodesk.Revit.DB;
using Autodesk.Revit.Utility;

namespace CustomExporterXml
{
  /// <summary>
  /// Out custom export context
  /// </summary>
  /// <remarks>
  /// An instance of this class is given to a custom exporter in Revit,
  /// which will then redirect all rendering output back to method of this class.
  /// </remarks>
  class ExportContext : IExportContext
  {
    // private data members

    private Document m_doc = null;
    private string m_filename = null;
    private XmlWriter m_writer = null;
    private Materials m_materials = new Materials();
    private Transformations m_transforms = new Transformations();
    private ExportOptions m_exportOptions = null;
    private Stopwatch m_stopwatch = null;


    /// <summary>
    ///  Constructor
    /// </summary>
    /// <param name="document">Document of which view are going to be exported</param>
    /// <param name="filename">Target filename to export to - overwrite if exists</param>
    /// <param name="options">Export options the user chose.</param>
    public ExportContext( Document document, string filename, ExportOptions options )
    {
      // Normally, there should be validation of the arguments,
      // but we'll just store them, for the sake of simplicity.
      // The export does not start yet. It will later when we get
      // the Start method invoked.
      m_doc = document;
      m_filename = filename;
      m_exportOptions = options;
    }

    /// <summary>
    /// A method for dumping some memory info, for performance tracking
    /// </summary>
    internal void WriteMemorySnapshot()
    {
      if( m_writer != null )
      {
        Process process = Process.GetCurrentProcess();
        if( process != null )
        {
          long memsize = process.PrivateMemorySize64 >> 10;
          m_writer.WriteComment( "Private memory: " + memsize.ToString( "n" ) );
          memsize = process.WorkingSet64 >> 10;
          m_writer.WriteComment( "Working set: " + memsize.ToString( "n" ) );
          memsize = process.PeakWorkingSet64 >> 10;
          m_writer.WriteComment( "Peak working set: " + memsize.ToString( "n" ) );
        }
      }
    }

    #region Implementation of the IExportContext interface methods

    public bool Start()
    {
      XmlTextWriter writer = new XmlTextWriter( m_filename, Encoding.UTF8 );
      writer.Formatting = Formatting.Indented;
      writer.Indentation = 2;

      m_writer = writer;

      m_writer.WriteStartDocument( false );
      m_writer.WriteStartElement( "Document" );
      m_writer.WriteAttributeString( "Title", m_doc.Title );

      // Some info for performance tuning

      WriteMemorySnapshot();
      m_stopwatch = new Stopwatch();
      m_stopwatch.Start();

      return true;   // yes, I am ready; please continue
    }

    public void Finish()
    {
      if( m_writer != null )
      {
        m_stopwatch.Stop();

        m_writer.WriteComment( "Elapsed time: " + m_stopwatch.Elapsed.TotalSeconds.ToString() );
        WriteMemorySnapshot();

        m_writer.WriteEndElement();
        m_writer.WriteEndDocument();
        m_writer.Flush();
        m_writer.Close();
        m_writer = null;
      }
    }

    public bool IsCanceled()
    {
      return false;  // in this sample, we never cancel
    }

    public RenderNodeAction OnViewBegin( ViewNode node )
    {
      m_writer.WriteStartElement( "View" );
      m_writer.WriteAttributeString( "Id", node.ViewId.IntegerValue.ToString() );

      View3D view = m_doc.GetElement( node.ViewId ) as View3D;
      if( view != null )
      {
        m_writer.WriteAttributeString( "Name", view.Name );
      }

      // technically, camera info is not always necessary,
      // but some clients really needed (renders, particularly)

      CameraInfo camera = node.GetCameraInfo();
      if( camera != null )
      {
        m_writer.WriteStartElement( "CameraInfo" );
        m_writer.WriteAttributeString( "IsPespective", camera.IsPespective.ToString() );
        m_writer.WriteAttributeString( "HorizontalExtent", camera.HorizontalExtent.ToString() );
        m_writer.WriteAttributeString( "VerticalExtent", camera.VerticalExtent.ToString() );
        m_writer.WriteAttributeString( "RightOffset", camera.RightOffset.ToString() );
        m_writer.WriteAttributeString( "UpOffset", camera.UpOffset.ToString() );
        m_writer.WriteAttributeString( "NearDistance", camera.NearDistance.ToString() );
        m_writer.WriteAttributeString( "FarDistance", camera.FarDistance.ToString() );
        if( camera.IsPespective )
        {
          m_writer.WriteAttributeString( "TargetDistance", camera.TargetDistance.ToString() );
        }
        m_writer.WriteEndElement();
      }

      // Setting our default LoD for the view
      // The scale goes from 0 to 10, but the value close to the edges
      // aren't really that usable, except maybe of experimenting

      node.LevelOfDetail = 5;   // a good middle ground 

      return RenderNodeAction.Proceed;
    }

    public void OnViewEnd( ElementId elementId )
    {
      m_writer.WriteEndElement();
      m_transforms.ClearTransforms();  // reset our transform stack (ought to be clear, but just in case)
    }

    public RenderNodeAction OnElementBegin( ElementId elementId )
    {
      // naturally, more info could be dumped at this level,
      // but we try to keep to keep the output simple

      m_writer.WriteStartElement( "Element" );
      m_writer.WriteAttributeString( "Id", elementId.IntegerValue.ToString() );

      // if we proceed, we get everything that belongs to the element

      return RenderNodeAction.Proceed;
    }

    public void OnElementEnd( ElementId elementId )
    {
      m_writer.WriteEndElement();
      m_materials.ResetCurrent();
    }

    public RenderNodeAction OnInstanceBegin( InstanceNode node )
    {
      m_writer.WriteStartElement( "Instance" );
      m_writer.WriteAttributeString( "SymbolId", node.GetSymbolId().ToString() );
      ElementType type = m_doc.GetElement( node.GetSymbolId() ) as ElementType;
      if( type != null )
      {
        m_writer.WriteAttributeString( "Category", type.Name );
      }
      m_writer.WriteElementString( "Transform", StringConversions.ValueToString( node.GetTransform() ) );

      // we need the transform to compute nested instances

      m_transforms.PushTransform( node.GetTransform() );

      if( !m_exportOptions.IncludeInstances )
      {
        return RenderNodeAction.Skip;
      }
      return RenderNodeAction.Proceed;
    }

    public void OnInstanceEnd( InstanceNode node )
    {
      m_writer.WriteEndElement();
      m_transforms.PopTransform();
    }

    public RenderNodeAction OnLinkBegin( LinkNode node )
    {
      m_writer.WriteStartElement( "Link" );
      m_writer.WriteAttributeString( "SymbolId", node.GetSymbolId().ToString() );

      m_writer.WriteElementString( "Transform", StringConversions.ValueToString( node.GetTransform() ) );

      Document linkdoc = node.GetDocument();
      if( linkdoc != null )
      {
        m_writer.WriteElementString( "Document", linkdoc.Title );
      }

      // like we instances, we may need to remember the link's transform

      m_transforms.PushTransform( node.GetTransform() );

      // we may or may not want to include links

      if( !m_exportOptions.IncludeLinks )
      {
        return RenderNodeAction.Skip;
      }
      return RenderNodeAction.Proceed;
    }

    public void OnLinkEnd( LinkNode node )
    {
      m_writer.WriteEndElement();
      m_transforms.PopTransform();
    }

    public RenderNodeAction OnFaceBegin( FaceNode node )
    {
      // remember, the exporter notifies about faces only
      // when it was requested at the time the export process started.
      // If it was not requested, the context would receive tessellated
      // meshes only, but not faces. Otherwise, both woudl be received
      // and it woudl be up to this context here to use what is needed.

      if( m_exportOptions.IncludeFaces )
      {
        m_writer.WriteStartElement( "Face" );

        Face face = node.GetFace();
        if( face != null )
        {
          m_writer.WriteAttributeString( "Id", face.GetHashCode().ToString() );

          string facetype = "Generic";
          if( null != face as ConicalFace )
          {
            facetype = "Conical";
          }
          else if( null != face as CylindricalFace )
          {
            facetype = "Cylindrical";
          }
          else if( null != face as HermiteFace )
          {
            facetype = "Hermite";
          }
          else if( null != face as PlanarFace )
          {
            facetype = "Planar";
          }
          else if( null != face as RevolvedFace )
          {
            facetype = "Revolved";
          }
          else if( null != face as RuledFace )
          {
            facetype = "Ruled";
          }

          m_writer.WriteAttributeString( "Shape", facetype );

          m_writer.WriteAttributeString( "TwoSided", face.IsTwoSided.ToString() );

          if( face.HasRegions )
          {
            m_writer.WriteAttributeString( "Regions", face.GetRegions().Count.ToString() );
          }

          m_writer.WriteStartElement( "Loops" );
          m_writer.WriteAttributeString( "Count", face.EdgeLoops.Size.ToString() );

          foreach( EdgeArray edgeloop in face.EdgeLoops )
          {
            m_writer.WriteStartElement( "Edges" );
            m_writer.WriteAttributeString( "Count", edgeloop.Size.ToString() );
            foreach( Edge edge in edgeloop )
            {
              m_writer.WriteStartElement( "Edge" );
              m_writer.WriteAttributeString( "Id", edge.GetHashCode().ToString() );
              m_writer.WriteAttributeString( "uv0", edge.EvaluateOnFace( 0, face ).ToString() );
              m_writer.WriteAttributeString( "uv1", edge.EvaluateOnFace( 1, face ).ToString() );

              Face neighborFace = null;
              Face nf = edge.GetFace( 0 );
              if( nf != null && nf != face )
              {
                neighborFace = nf;
              }
              else
              {
                nf = edge.GetFace( 1 );
                if( nf != null && nf != face )
                {
                  neighborFace = nf;
                }
              }

              if( neighborFace != null )
              {
                m_writer.WriteAttributeString( "Neighbor_face", neighborFace.GetHashCode().ToString() );
              }

              m_writer.WriteEndElement();
            }
            m_writer.WriteEndElement();
          }
          m_writer.WriteEndElement();
        }
      }

      // if we request skipping this face, we won't get its meshes

      if( !m_exportOptions.IncludeMeshes )
      {
        return RenderNodeAction.Skip;
      }
      return RenderNodeAction.Proceed;
    }

    public void OnFaceEnd( FaceNode node )
    {
      if( m_exportOptions.IncludeFaces )
      {
        m_writer.WriteEndElement();
      }
    }

    public void OnLight( LightNode node )
    {
      // More info about lights can be acquired here using the standard Light API

      m_writer.WriteStartElement( "Light" );
      m_writer.WriteElementString( "Transform", StringConversions.ValueToString( node.GetTransform() ) );
      Asset asset = node.GetAsset();
      if( asset != null )
      {
        // Assets are really more for Revit use internal uses,
        // but it is possible to get them for lights.
        ExportAsset.WriteAssetElement( m_writer, asset );
      }
      m_writer.WriteEndElement();
      return;
    }

    public void OnRPC( RPCNode node )
    {
      // RPCs cannot get (due to copyrights) any info besides assets.
      // There is currently no public API for RPCs

      m_writer.WriteStartElement( "RPC" );
      m_writer.WriteElementString( "Transform", StringConversions.ValueToString( node.GetTransform() ) );
      Asset asset = node.GetAsset();
      if( asset != null )
      {
        ExportAsset.WriteAssetElement( m_writer, asset );
      }
      m_writer.WriteEndElement();
      return;
    }

    public void OnDaylightPortal( DaylightPortalNode node )
    {
      // Like RPCs, Daylight Portals too have their assets available only.
      // THere is no other public API for them currently available.

      m_writer.WriteStartElement( "DaylightPortal" );
      m_writer.WriteElementString( "Transform", StringConversions.ValueToString( node.GetTransform() ) );
      Asset asset = node.GetAsset();
      if( asset != null )
      {
        ExportAsset.WriteAssetElement( m_writer, asset );
      }
      m_writer.WriteEndElement();
      return;
    }

    public void OnMaterial( MaterialNode node )
    {
      // In this sample, we only export the current material once

      if( m_materials.IsCurrent( node.MaterialId ) )
      {
        return;
      }

      bool isNew = m_materials.Add( node.MaterialId );

      m_writer.WriteStartElement( "Material" );
      {
        m_writer.WriteAttributeString( "Id", node.MaterialId.ToString() );

        if( node.MaterialId == ElementId.InvalidElementId )
        {
          m_writer.WriteAttributeString( "Name", "default" );
        }
        else
        {
          Material material = m_doc.GetElement( node.MaterialId ) as Material;
          if( material != null )
          {
            m_writer.WriteAttributeString( "Class", material.MaterialClass );
            m_writer.WriteAttributeString( "Name", material.Name );
          }
        }

        if( isNew && m_exportOptions.IncludeMaterials )
        {
          m_writer.WriteElementString( "Transparency", node.Transparency.ToString() );
          m_writer.WriteElementString( "Smoothness", node.Smoothness.ToString() );
          m_writer.WriteElementString( "Glossiness", node.Glossiness.ToString() );
          m_writer.WriteElementString( "Overridden", node.HasOverriddenAppearance.ToString() );

          // A lot of attributes can be obtained from Material API in Revit,
          // but dumping the material's asset is also possible, even though
          // assets are really for Revit internal use and their format and content
          // is not documented.

          // Apperarence can be overrided in some situations; for example
          // when a decal is applied to a face and gets rendered with the 
          // material underneath it.

          if( node.HasOverriddenAppearance )
          {
            ExportAsset.WriteAssetElement( m_writer, node.GetAppearanceOverride() );
          }
          else
          {
            ExportAsset.WriteAssetElement( m_writer, node.GetAppearance() );
          }
        }
      }
      m_writer.WriteEndElement();

      return;
    }

    public void OnPolymesh( PolymeshTopology node )
    {
      // A polymesh is the very bottom node in the hierarchy of rendering a model.
      // Not every exporter may need need polymeshes; on the other hand, graphic
      // renderer and exporter alike will mostly need nothing but polymeshes.

      m_writer.WriteStartElement( "Polymesh" );
      m_writer.WriteAttributeString( "Facets", node.NumberOfFacets.ToString() );
      m_writer.WriteAttributeString( "Points", node.NumberOfPoints.ToString() );
      m_writer.WriteAttributeString( "UVs", node.NumberOfUVs.ToString() );
      m_writer.WriteAttributeString( "Normals", node.NumberOfNormals.ToString() );

      m_writer.WriteElementString( "Facets", StringConversions.ValueToString( node.GetFacets() ) );

      if( m_exportOptions.ApplyTransforms )   // we want either raw or transformed coordinates
      {
        m_writer.WriteElementString( "Points", StringConversions.ValueToString( m_transforms.ApplyTransform( node.GetPoints() ) ) );
        m_writer.WriteElementString( "Normals", StringConversions.ValueToString( m_transforms.ApplyTransform( node.GetNormals() ) ) );
      }
      else
      {
        m_writer.WriteElementString( "Points", StringConversions.ValueToString( node.GetPoints() ) );
        m_writer.WriteElementString( "Normals", StringConversions.ValueToString( node.GetNormals() ) );

      }

      if( node.NumberOfUVs > 0 )
      {
        m_writer.WriteElementString( "UVs", StringConversions.ValueToString( node.GetUVs() ) );
      }

      m_writer.WriteEndElement();

      return;
    }

    #endregion

  }  // class

}  // namespace
