﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;

namespace CustomExporterXml
{
  /// <summary>
  /// A simple dialog to gathering user export options.
  /// </summary>
  public partial class ExportOptionsDialog : Form
  {
    private ExportOptions m_options = null;
    private string m_filepath = null;

    /// <summary>
    /// COnstructor
    /// </summary>
    public ExportOptionsDialog(
      string defaultPath,
      IEnumerable<string> viewnames )
    {
      InitializeComponent();

      m_filepath = defaultPath;
      if( !string.IsNullOrEmpty( defaultPath ) )
      {
        this.txtOutputPath.Text = m_filepath;
      }

      foreach( string name in viewnames )
      {
        this.lstViews.Items.Add( name );
      }
    }

    /// <summary>
    /// A getter of acquired options
    /// </summary>
    public ExportOptions GetOptions()
    {
      return ( m_options != null )
        ? new ExportOptions( m_options )
        : null;
    }

    /// <summary>
    /// A getter for the output path
    /// </summary>
    public string GetFilePath()
    {
      return m_filepath;
    }

    /// <summary>
    /// A getter for the list of selected views to export.
    /// </summary>
    public IList<int> GetCheckedViewsIndices()
    {
      IList<int> indices = new List<int>( lstViews.CheckedIndices.Count );
      foreach( int index in lstViews.CheckedIndices )
      {
        indices.Add( index );
      }
      return indices;
    }


    /// <summary>
    /// Action when hitting the OK button
    /// </summary>
    private void btnOK_Click( object sender, EventArgs e )
    {
      m_options = new ExportOptions();

      m_options.IncludeInstances = this.chkInstances.Checked;
      m_options.IncludeLinks = this.chkLinks.Checked;
      m_options.IncludeFaces = this.chkFaces.Checked;
      m_options.IncludeMeshes = this.chkMeshes.Checked;
      m_options.IncludeMaterials = this.chkMaterial.Checked;
      m_options.ApplyTransforms = this.chkApplyTransforms.Checked;

      m_filepath = this.txtOutputPath.Text;

      this.DialogResult = DialogResult.OK;
      Close();
    }


    /// <summary>
    /// Action when browsing for an output file
    /// </summary>
    private void btnBrowse_Click( object sender, EventArgs e )
    {
      string newPath = SelectOutputFile( m_filepath );
      if( !string.IsNullOrEmpty( newPath ) )
      {
        m_filepath = newPath;
        this.txtOutputPath.Text = m_filepath;
      }
    }


    /// <summary>
    /// Customized file-save procedure
    /// </summary>
    /// <param name="defaultname"></param>
    /// <returns></returns>
    private string SelectOutputFile( string defaultname )
    {
      if( string.IsNullOrEmpty( defaultname ) )
      {
        defaultname = "VisOutput";
      }
      var sfd = new SaveFileDialog();
      sfd.Title = "Select output file";
      sfd.FileName = defaultname;
      sfd.DefaultExt = "xml";
      sfd.Filter = "XML files|*.xml|All files|*.*";
      sfd.CheckFileExists = false;
      sfd.CheckPathExists = true;
      sfd.AddExtension = true;
      sfd.OverwritePrompt = true;
      sfd.InitialDirectory = Environment.GetFolderPath(
        Environment.SpecialFolder.MyDocuments );

      DialogResult result = sfd.ShowDialog();
      if( DialogResult.OK == result )
      {
        return sfd.FileName;
      }
      return null;
    }

    /// <summary>
    /// Procedure for checking all output options On or Off
    /// </summary>
    private void SelectCheckboxes( bool value )
    {
      this.chkInstances.Checked = value;
      this.chkLinks.Checked = value;
      this.chkFaces.Checked = value;
      this.chkMeshes.Checked = value;
      this.chkMaterial.Checked = value;
      this.chkApplyTransforms.Checked = value;
    }

    /// <summary>
    /// Checking On all options
    /// </summary>
    private void btnSelectAll_Click( object sender, EventArgs e )
    {
      SelectCheckboxes( true );
    }

    /// <summary>
    /// Checking Off all options
    /// </summary>
    private void btnSelectNone_Click( object sender, EventArgs e )
    {
      SelectCheckboxes( false );
    }

    /// <summary>
    /// Selecting ALL available views
    /// </summary>
    private void btnAllViews_Click( object sender, EventArgs e )
    {
      for( int idx = 0; idx < this.lstViews.Items.Count; idx++ )
      {
        this.lstViews.SetItemCheckState( idx, CheckState.Checked );
      }
    }

    /// <summary>
    /// Deselecting ALL available views
    /// </summary>
    private void btnNoViews_Click( object sender, EventArgs e )
    {
      for( int idx = 0; idx < this.lstViews.Items.Count; idx++ )
      {
        this.lstViews.SetItemCheckState( idx, CheckState.Unchecked );
      }
    }
  }
}
