﻿using System;
using System.Collections.Generic;

using Autodesk.Revit.DB;

namespace CustomExporterXml
{
  /// <summary>
  /// A helper class for caching stacked transformations
  /// </summary>
  class Transformations
  {
    // A stack of output transformations.
    // The stack contains combined transformations, not individual transforms

    private Stack<Transform> m_transforms = new Stack<Transform>();
    private Transform m_currentTransform = Transform.Identity;

    // adding a transform on the stack
    public void PushTransform( Transform t )
    {
      // the current transform is added to the stack
      m_transforms.Push( m_currentTransform );
      // and then a new current transform is calculated by combining with the given one
      m_currentTransform = m_currentTransform.Multiply( t );
    }

    // removing one transform from the stack
    public void PopTransform()
    {
      if( m_transforms.Count > 0 )
      {
        // the new current transform will be the one removed from top of the stack
        m_currentTransform = m_transforms.Pop();
      }
      else
      {
        // this situation would deserve a debug warning
      }
    }

    // clearing the stack
    public void ClearTransforms()
    {
      m_transforms.Clear();
      m_currentTransform = Transform.Identity;
    }

    // applying the current transform to a collection of points
    public IList<XYZ> ApplyTransform( IList<XYZ> points )
    {
      IList<XYZ> newPoints = new List<XYZ>( points.Count );

      foreach( XYZ xyz in points )
      {
        newPoints.Add( m_currentTransform.OfPoint( xyz ) );
      }

      return newPoints;
    }
  }  // class Transformations
}
