#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.ExtensibleStorage;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

namespace DataStorageSample2
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    private readonly Guid dsSectionId =
        new Guid( "{E9340A48-105D-4809-BB30-978E4F305D20}" );

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      var sections =
          GetProjectSections( doc )
              .ToList();

      // Access current selection
      Selection sel = uidoc.Selection;

      var pb = sel.PickBox( PickBoxStyle.Crossing );

      Section newSection = new Section();
      newSection.Name = "Section " + ( sections.Count + 1 ).ToString();
      newSection.BoundingBoxMin = pb.Min;
      newSection.BoundingBoxMax = pb.Max;
      newSection.View = doc.ActiveView;

      sections.Add( newSection );

      using( Transaction t = new Transaction( doc, "Add section" ) )
      {
        t.Start();

        //Create section entity    
        Entity sectionEntity = new Entity( SectionSchema.GetSchema() );
        sectionEntity.Set( "Name", newSection.Name );
        sectionEntity.Set( "BoundingBoxMin", newSection.BoundingBoxMin, DisplayUnitType.DUT_CUSTOM );
        sectionEntity.Set( "BoundingBoxMax", newSection.BoundingBoxMax, DisplayUnitType.DUT_CUSTOM );
        sectionEntity.Set( "ViewId", newSection.View.Id );

        // create identity entity to identify section in the future
        Entity idEntity = new Entity( DataStorageUniqueIdSchema.GetSchema() );
        idEntity.Set( "Id", dsSectionId );

        //Create a data storage and set entities

        DataStorage sectionDS = DataStorage.Create( doc );
        sectionDS.SetEntity( idEntity );
        sectionDS.SetEntity( sectionEntity );

        t.Commit();
      }


      StringBuilder sb = new StringBuilder();
      sb.AppendLine( "Sections:" );
      foreach( var section in sections )
      {
        sb.AppendFormat( "Name: {0} Min: {1:F2}; {2:F2}; {3:F2}; Max: {4:F2}; {5:F2};{6:F2}; View: {7}\r\n",
            section.Name,
            section.BoundingBoxMin.X, section.BoundingBoxMin.Y, section.BoundingBoxMin.Z,
            section.BoundingBoxMax.X, section.BoundingBoxMax.Y, section.BoundingBoxMax.Z,
            section.View == null ? string.Empty : section.View.Name );
      }

      TaskDialog.Show( "Project sections", sb.ToString() );

      return Result.Succeeded;
    }


    IEnumerable<Section> GetProjectSections( Document doc )
    {
      FilteredElementCollector collector =
          new FilteredElementCollector( doc );

      var dataStorages =
          collector
              .OfClass( typeof( DataStorage ) );

      foreach( var dataStorage in dataStorages )
      {
        Entity dsIdEntity =
            dataStorage.GetEntity( DataStorageUniqueIdSchema.GetSchema() );

        if( !dsIdEntity.IsValid() )
          continue;

        var id = dsIdEntity.Get<Guid>( "Id" );
        if( id.Equals( dsSectionId ) )
        {
          Entity sectionEntity =
              dataStorage.GetEntity( SectionSchema.GetSchema() );
          if( !sectionEntity.IsValid() )
            continue;

          Section section = new Section();
          section.Name = sectionEntity.Get<string>( "Name" );
          section.BoundingBoxMin =
              sectionEntity.Get<XYZ>( "BoundingBoxMin", DisplayUnitType.DUT_CUSTOM );
          section.BoundingBoxMax =
              sectionEntity.Get<XYZ>( "BoundingBoxMax", DisplayUnitType.DUT_CUSTOM );
          var viewId = sectionEntity.Get<ElementId>( "ViewId" );
          if( viewId != ElementId.InvalidElementId )
            section.View = doc.GetElement( viewId ) as View;

          yield return section;
        }
      }
    }
  }
}
