#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
#endregion

namespace DeleteUnnamedNonHostingReferencePlanes
{
  /// <summary>
  /// Delete all reference planes that have not been named and are not hosting any elements.
  /// In other words, check whether the reference plane has been named.
  /// If not, check whether it hosts any elements.
  /// If not, delete it.
  /// Actually, to check whether it hosts any elements, we delete it anyway:
  /// 314_object_relationship.htm
  /// 319_object_relationship_vb.htm
  /// The deletion returns the number of elements deleted.
  /// If it is greater than one (the ref plane itself), it hosted something.
  /// In that case, abort the transaction and do not delete.
  /// </summary>
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    static int _i = 0;

    bool DeleteIfNotHosting( ReferencePlane rp )
    {
      bool rc = false;

      Document doc = rp.Document;

      Transaction tx = new Transaction( doc );

      tx.Start( "Delete ReferencePlane " 
        + (++_i).ToString() );

      // Deletion simply fails if the reference plane 
      // hosts anything. If so, the return value ids 
      // is null:

      ICollection<ElementId> ids = doc.Delete( rp );

      if( null == ids || 1 < ids.Count )
      {
        tx.RollBack();
      }
      else
      {
        tx.Commit();
        rc = true;
      }
      return rc;
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      // Construct a parameter filter to get only 
      // unnamed reference planes, i.e. reference 
      // planes whose name equals the empty string:

      BuiltInParameter bip
        = BuiltInParameter.DATUM_TEXT;

      ParameterValueProvider provider
        = new ParameterValueProvider(
          new ElementId( bip ) );

      FilterStringRuleEvaluator evaluator 
        = new FilterStringEquals();

      FilterStringRule rule = new FilterStringRule(
        provider, evaluator, "", false );

      ElementParameterFilter filter
        = new ElementParameterFilter( rule );

      FilteredElementCollector col 
        = new FilteredElementCollector( doc )
          .OfClass( typeof( ReferencePlane ) )
          .WherePasses( filter );

      int n = 0;
      int nDeleted = 0;

      // No need to cast ... this is pretty nifty,
      // I find ... grab the elements as ReferencePlane
      // instances, since the filter guarantees that 
      // only ReferencePlane instances are selected.

      foreach( ReferencePlane rp in col )
      {
        ++n;
        nDeleted += DeleteIfNotHosting( rp ) ? 1 : 0;
      }

      Debug.Print( 
        "{0} unnamed reference plane{1} examined, {2} of them were deleted.",
        n, (1==n?"":"s"), nDeleted );

      return Result.Succeeded;
    }
  }
}
