#region Namespaces
using System;
using System.Linq;
using System.Collections.Generic;
using System.Diagnostics;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

namespace DirectShapeMinSize
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    /// <summary>
    /// Set this to true to iterate through smaller 
    /// and smaller tetrahedron sizes until we hit
    /// Revit's precision limit.
    /// </summary>
    static bool Iterate_until_crash = false;

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      // Find GraphicsStyle

      FilteredElementCollector collector
        = new FilteredElementCollector( doc )
          .OfClass( typeof( GraphicsStyle ) );

      GraphicsStyle style = collector.Cast<GraphicsStyle>()
        .FirstOrDefault<GraphicsStyle>( gs => gs.Name.Equals( "<Sketch>" ) );

      ElementId graphicsStyleId = null;

      if( style != null )
      {
        graphicsStyleId = style.Id;
      }

      // Modify document within a transaction

      using( Transaction tx = new Transaction( doc ) )
      {
        tx.Start( "Create DirectShape" );

        double length = 1; // foot

        try
        {
          do
          {
            length = 0.5 * length;

            Debug.Print(
                "Creating DirectShape tetrahedron with side length {0}...",
                length );

            List<XYZ> args = new List<XYZ>( 3 );

            TessellatedShapeBuilder builder = new TessellatedShapeBuilder();

            builder.OpenConnectedFaceSet( false );

            args.Add( XYZ.Zero );
            args.Add( length * XYZ.BasisX );
            args.Add( length * XYZ.BasisY );
            builder.AddFace( new TessellatedFace( args, ElementId.InvalidElementId ) );

            args.Clear();
            args.Add( XYZ.Zero );
            args.Add( length * XYZ.BasisX );
            args.Add( length * XYZ.BasisZ );
            builder.AddFace( new TessellatedFace( args, ElementId.InvalidElementId ) );

            args.Clear();
            args.Add( XYZ.Zero );
            args.Add( length * XYZ.BasisY );
            args.Add( length * XYZ.BasisZ );
            builder.AddFace( new TessellatedFace( args, ElementId.InvalidElementId ) );

            args.Clear();
            args.Add( length * XYZ.BasisX );
            args.Add( length * XYZ.BasisY );
            args.Add( length * XYZ.BasisZ );
            builder.AddFace( new TessellatedFace( args, ElementId.InvalidElementId ) );

            builder.CloseConnectedFaceSet();

            TessellatedShapeBuilderResult result
              = builder.Build(
                TessellatedShapeBuilderTarget.Solid,
                TessellatedShapeBuilderFallback.Abort,
                graphicsStyleId );

            // Pre-release code from DevDays

            //DirectShape ds = DirectShape.CreateElement(
            //  doc, result.GetGeometricalObjects(), "A", "B");

            //ds.SetCategoryId(new ElementId(
            //  BuiltInCategory.OST_GenericModel));

            // Code updated for Revit UR1

            ElementId categoryId = new ElementId(
              BuiltInCategory.OST_GenericModel );

            DirectShape ds = DirectShape.CreateElement(
              doc, categoryId, "A", "B" );

            ds.SetShape( result.GetGeometricalObjects() );

            ds.Name = "Test";
          }
          while( Iterate_until_crash && 0 < length );
        }
        catch( Exception e )
        {
          Debug.Print(
            "Creating DirectShape tetrahedron with side length {0} "
            + "threw exception '{1}'",
            length, e.Message );

          message = e.Message;
          return Result.Failed;
        }
        tx.Commit();
      }
      return Result.Succeeded;
    }
  }
}
