#region Namespaces
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

namespace DisplayBoundary
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      FilteredElementCollector col = null;

      // Access current selection

      ElementSet set = uidoc.Selection.Elements;

      int n = set.Size;

      if( 0 < n )
      {
        //foreach( Element e in sel.Elements )
        //{
        //  if( e is Area )
        //  {
        //    ids.Add( e.Id );
        //  }
        //}

        //List<ElementId> ids = new List<ElementId>( sel.Elements.Select<Element,ElementId>( e => e.Id ) );

        List<ElementId> ids 
          = new List<ElementId>( set
            .OfType<Area>()
            .Select<Area,ElementId>( e => e.Id ) );

        if( 0 == ids.Count )
        {
          message = "Please select some area alements "
            + " before running his command, or nothing "
            + "at all to process all of them.";

          return Result.Failed;
        }

        // When instantiating a filtered element 
        // collector using preselected elements, 
        // you still must apply a filter, even if 
        // it has no effect, or the collector will 
        // throw an exception when accessing its
        // elements:

        col = new FilteredElementCollector( doc, ids )
          .WhereElementIsNotElementType();
      }
      else
      {
        // Retrieve all area elements from database.
        // We cannot use the OfClass filter with the 
        // Area class, since it throws an exception 
        // saying "Input type (Autodesk.Revit.DB.Area)
        // is of an element type that exists in the 
        // API, but not in Revit's native object model. 
        // Try using Autodesk.Revit.DB.SpatialElement 
        // instead, and then postprocessing the results 
        // to find the elements of interest.

        col = new FilteredElementCollector( doc )
          .OfClass( typeof( SpatialElement ) );
      }

      // Specify boundary options

      SpatialElementBoundaryOptions opt 
        = new SpatialElementBoundaryOptions();

      opt.StoreFreeBoundaryFaces = true;

      opt.SpatialElementBoundaryLocation =
        SpatialElementBoundaryLocation.Center; // loops closed
        //SpatialElementBoundaryLocation.Finish; // loops not closed

      using( Transaction tx = new Transaction( doc ) )
      {
        tx.Start( "Convert Area Loops To Model Curves" );

        // Filtered element collector is iterable

        foreach( SpatialElement e in col )
        {
          Area area = e as Area;

          if( null != area )
          {
            Debug.Print( area.Name );

            double z = area.Level.Elevation;

            Plane levelPlane = app.Create.NewPlane(
              XYZ.BasisZ, new XYZ( 0, 0, z ) );

            SketchPlane sketchPlane
              = doc.Create.NewSketchPlane( levelPlane );

            //sketchPlane.Name = "Model Curves for Area "
            //  + area.Id.ToString();

            IList<IList<BoundarySegment>> loops
              = area.GetBoundarySegments( opt );

            foreach( IList<BoundarySegment> L in loops )
            {
              foreach( BoundarySegment seg in L )
              {
                doc.Create.NewModelCurve(
                  seg.Curve, sketchPlane );
              }
            }
          }
        }

        tx.Commit();
      }
      return Result.Succeeded;
    }
  }
}
