#region Namespaces
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.ExtensibleStorage;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

namespace DivideParts
{
  [Transaction( TransactionMode.Manual )]
  public class BuildPanelsCommand : IExternalCommand
  {
    static double DegreesToRadians( double r )
    {
      return ( r / Math.PI ) * 180;
    }

    /// <summary>
    /// A selection filter for wall elements.
    /// </summary>
    class WallSelectionFilter : ISelectionFilter
    {
      public bool AllowElement( Element e )
      {
        return e is Wall;
      }

      public bool AllowReference( Reference r, XYZ p )
      {
        return true;
      }
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      try
      {
        if( null == commandData )
        {
          throw new ArgumentNullException( "commandData" );
        }

        UIApplication uiapp = commandData.Application;
        Application app = uiapp.Application;
        UIDocument uidoc = uiapp.ActiveUIDocument;
        Document doc = uidoc.Document;

        Reference r = null;

        try
        {
          r = uidoc.Selection.PickObject(
            ObjectType.Element,
            new WallSelectionFilter(),
            "Select a wall to split into panels" );
        }
        catch( Autodesk.Revit.Exceptions
          .OperationCanceledException )
        {
          return Result.Cancelled;
        }

        Wall wall = ( r == null || r.ElementId
            == ElementId.InvalidElementId )
          ? null
          : doc.GetElement( r.ElementId ) as Wall;

        if( wall == null )
        {
          message = "Unable to retrieve wall.";
          return Result.Failed;
        }

        LocationCurve location
          = wall.Location as LocationCurve;

        if( null == location )
        {
          message = "Unable to retrieve wall location curve.";
          return Result.Failed;
        }

        Line line = location.Curve as Line;

        if( null == location )
        {
          message = "Unable to retrieve wall location line.";
          return Result.Failed;
        }

        using( Transaction transaction = new Transaction( doc ) )
        {
          transaction.Start( "Building panels" );

          IList<ElementId> wallList = new List<ElementId>( 1 );

          wallList.Add( r.ElementId );

          if( PartUtils.AreElementsValidForCreateParts( 
            doc, wallList ) )
          {
            PartUtils.CreateParts( doc, wallList );
            
            doc.Regenerate();

            ICollection<ElementId> parts 
              = PartUtils.GetAssociatedParts( 
                doc, wall.Id, false, false );

            if( PartUtils.ArePartsValidForDivide( 
              doc, parts ) )
            {
              int divisions = 5;

              //IList<XYZ> divisionPoints = new List<XYZ>();

              XYZ origin = line.Origin;

              XYZ delta = line.Direction.Multiply(
                line.Length / divisions );

              Transform shiftDelta
                = Transform.get_Translation( delta );

              // Construct a 90 degree rotation in the 
              // XY plane around the line start point

              Transform rotation = Transform.get_Rotation(
                origin, XYZ.BasisZ, 0.5 * Math.PI );

              // A vector perpendicular to the wall with
              // length equal to twice the wall width

              XYZ wallWidthVector = rotation.OfVector(
                line.Direction.Multiply( 2 * wall.Width ) );

              Curve intersectionLine
                = app.Create.NewLineBound( // Line.CreateBound
                  origin + wallWidthVector,
                  origin - wallWidthVector );

              //for( int i = 1; i < divisions; ++i )
              //{
              //  divisionPoints.Add( origin + i * delta );
              //}

              IList<Curve> curveArray = new List<Curve>();

              for( int i = 1; i < divisions; ++i )
              {
                // Create a line for each division point
                // Start point at (division point - wall width vector)
                // End point at (division point + wall width vector) 

                //Line intersectionLine = Line.CreateBound( 
                //  divisionPoints[i].Add( wallWidthVector.Negate() ),
                //  divisionPoints[i].Add( wallWidthVector ) );

                // Creating a 90deg rotation transform

                //Transform rotate90Deg = Transform.CreateRotationAtPoint( 
                //  XYZ.BasisZ,
                //  Units.ToRadians( 90 ),
                //  divisionPoints[i] );

                // Based on intersectionLines creating a 90deg rotated curve
                // This curve, perpendicular to the wall will be used to divide the object

                //Curve intersectionCurve = intersectionLine
                //  .CreateTransformed( rotate90Deg );

                intersectionLine = intersectionLine
                  .get_Transformed( shiftDelta );

                curveArray.Add( intersectionLine );
              }

              //SketchPlane divisionSketchPlane
              //  = SketchPlane.Create( doc,
              //    new Plane( XYZ.BasisZ, line.Origin ) );

              SketchPlane divisionSketchPlane
                = doc.Create.NewSketchPlane(
                  app.Create.NewPlane(
                    XYZ.BasisZ, line.Origin ) );

              // An empty list of intersecting ElementIds

              IList<ElementId> intersectionElementIds
                = new List<ElementId>();

              PartUtils.DivideParts( doc, parts, 
                intersectionElementIds, curveArray, 
                divisionSketchPlane.Id );
            }
            doc.ActiveView.PartsVisibility 
              = PartsVisibility.ShowPartsOnly;
          }
          transaction.Commit();
        }
        return Result.Succeeded;
      }
      catch( Exception e )
      {
        message = e.Message;
        return Result.Failed;
      }
    }

    #region Obsolete first version
    public Result Execute1(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      try
      {
        if( null == commandData )
        {
          throw new ArgumentNullException(
            "commandData" );
        }

        UIApplication uiapp = commandData.Application;
        Application app = uiapp.Application;
        UIDocument uidoc = uiapp.ActiveUIDocument;
        Document doc = uidoc.Document;

        using( Transaction tx = new Transaction( doc ) )
        {
          tx.Start( "Building panels" );

          ISelectionFilter wallFilter
            = new WallSelectionFilter();

          Reference r
            = uidoc.Selection.PickObject(
              ObjectType.Element, wallFilter,
              "Select a wall to split into panels" );

          if( r != null
            && r.ElementId
              != ElementId.InvalidElementId )
          {
            Wall wall = doc.GetElement(
              r.ElementId ) as Wall;

            if( wall == null )
            {
              return Result.Failed;
            }

            IList<ElementId> wallList
              = new List<ElementId>();

            wallList.Add( r.ElementId );

            LocationCurve lc
              = wall.Location as LocationCurve;

            Line line = lc.Curve as Line;

            if( null == line )
            {
              message = "Unable to retrieve wall "
                + "location line.";

              return Result.Failed;
            }

            int divisions = 5;

            IList<XYZ> divisionPoints
              = new List<XYZ>();

            IList<XYZ> intersectionLineOriginPoints
              = new List<XYZ>();

            IList<XYZ> intersectionLineEndPoints
              = new List<XYZ>();

            XYZ origin = line.Origin;

            XYZ v = line.Direction.Multiply(
              line.Length / divisions );

            for( int i = 1; i < divisions; ++i )
            {
              divisionPoints.Add( origin + i * v );
            }

            // Double the width of the wall

            XYZ wallWidthVector = line.Direction
              .Multiply( 2 * wall.Width );

            IList<ElementId> intersectionElementIds
              = new List<ElementId>();

            IList<Curve> curveArray = new List<Curve>();

            for( int i = 0; i < divisions - 1; ++i )
            {
              Line intersectionLine
                = app.Create.NewLineBound( // Line.CreateBound
                  divisionPoints[i].Add(
                    wallWidthVector.Negate() ),
                  divisionPoints[i].Add(
                    wallWidthVector ) );

              //Transform rotate90Deg 
              //  = Transform.CreateRotationAtPoint( 
              //  XYZ.BasisZ, Units.ToRadians( 90 ), 
              //  divisionPoints[i] );

              Transform rotate90Deg
                = Transform.get_Rotation(
                  divisionPoints[i], XYZ.BasisZ,
                  DegreesToRadians( 90 ) );

              //Curve intersectionCurve 
              //  = intersectionLine.CreateTransformed( 
              //    rotate90Deg );

              Curve intersectionCurve
                = intersectionLine.get_Transformed(
                  rotate90Deg );

              curveArray.Add( intersectionCurve );

              XYZ startPoint = intersectionCurve
                .get_EndPoint( 0 ); // GetEndPoint

              XYZ endPoint = intersectionCurve
                .get_EndPoint( 1 ); // GetEndPoint

              ReferencePlane intersectionPlane
                = doc.Create.NewReferencePlane2(
                  startPoint, endPoint,
                  line.Direction, doc.ActiveView );

              intersectionElementIds.Add(
                intersectionPlane.Id );
            }

            //SketchPlane divisionSketchPlane
            //  = SketchPlane.Create( doc,
            //    new Plane( XYZ.BasisZ, line.Origin ) );

            SketchPlane divisionSketchPlane
              = doc.Create.NewSketchPlane(
                app.Create.NewPlane(
                  XYZ.BasisZ, line.Origin ) );

            if( PartUtils.AreElementsValidForCreateParts(
              doc, wallList ) )
            {
              PartUtils.CreateParts( doc, wallList );

              doc.Regenerate();

              ICollection<ElementId> parts
                = PartUtils.GetAssociatedParts(
                  doc, wall.Id, false, false );

              if( PartUtils.ArePartsValidForDivide(
                doc, parts ) )
              {
                PartUtils.DivideParts( doc, parts,
                  intersectionElementIds, curveArray,
                  divisionSketchPlane.Id );
              }
              // doc.Regenerate();
            }
          }
          tx.Commit();
        }
        return Result.Succeeded;
      }
      catch( Exception e )
      {
        message = e.Message;
        return Result.Failed;
      }
    }
    #endregion // Obsolete first version

    #region Harry Mattison AU 2011 CP5211 sample code
    public class CreateAndDivideParts : IExternalCommand
    {
      public Result Execute(
        ExternalCommandData commandData,
        ref string message,
        ElementSet elements )
      {
        Autodesk.Revit.ApplicationServices.Application app = commandData.Application.Application;
        Document doc = commandData.Application.ActiveUIDocument.Document;
        UIDocument uidoc = commandData.Application.ActiveUIDocument;

        IList<ElementId> idList = new List<ElementId>();

        // Select an element to use for part creation. Can be a host element (floor, roof, etc) or a part created from one of these elements.
        Element pickedElement = doc.GetElement( uidoc.Selection.PickObject( ObjectType.Element ) );
        idList.Add( pickedElement.Id );

        // Determine which elements to divide.
        ICollection<ElementId> elementIdsToDivide = new List<ElementId>();
        if( PartUtils.AreElementsValidForCreateParts( doc, idList ) )
        {
          // AreElementsValidForCreateParts returned true, so the selected element is not a part but it is an element that can be used to create a part. 
          Transaction createPartTransaction = new Transaction( doc, "Create Part" );
          createPartTransaction.Start();
          PartUtils.CreateParts( doc, idList ); // create the parts
          createPartTransaction.Commit();

          elementIdsToDivide = PartUtils.GetAssociatedParts( doc, pickedElement.Id, true, true ); // get the id of the newly created part
        }
        else if( pickedElement is Part )
        {
          // The selected element is a part, so that part will be divided. 
          elementIdsToDivide.Add( pickedElement.Id );
        }

        // Create geometry that will be used to divide the part. For this example, a new part will be divided from the main part that is one quarter of the face. More complex intelligence could be coded to divide the part based on construction logistics or the properties of the materials being used to create the part.
        XYZ pointRight = null;
        XYZ pointTop = null;
        XYZ pointCorner = null;
        XYZ pointCenter = null;

        SketchPlane sketchPlane = null;

        Options opt = new Options();
        opt.ComputeReferences = true;
        GeometryElement geomElem = pickedElement.get_Geometry( opt );
        foreach( GeometryObject geomObject in geomElem.Objects )
        {
          if( geomObject is Solid ) // get the solid geometry of the selected element
          {
            Solid solid = geomObject as Solid;
            FaceArray faceArray = solid.Faces;
            foreach( Face face in faceArray )
            {
              // find the center of the face
              BoundingBoxUV bbox = face.GetBoundingBox();
              UV center = new UV( ( bbox.Max.U - bbox.Min.U ) / 2 + bbox.Min.U, ( bbox.Max.V - bbox.Min.V ) / 2 + bbox.Min.V );
              XYZ faceNormal = face.ComputeNormal( center );
              if( faceNormal.IsAlmostEqualTo( XYZ.BasisZ ) ) // this example is designed to work with a floor or other element with a large face whose normal is in the Z direction
              {
                Transaction sketchPlaneTransaction = new Transaction( doc, "Create Sketch Plane" );
                sketchPlaneTransaction.Start();
                sketchPlane = doc.Create.NewSketchPlane( face as PlanarFace );
                sketchPlaneTransaction.Commit();

                pointCenter = face.Evaluate( center );

                UV top = new UV( ( bbox.Max.U - bbox.Min.U ) / 2 + bbox.Min.U, bbox.Max.V );
                pointTop = face.Evaluate( top );

                UV right = new UV( bbox.Max.U, ( bbox.Max.V - bbox.Min.V ) / 2 + bbox.Min.V );
                pointRight = face.Evaluate( right );

                UV corner = new UV( bbox.Max.U, bbox.Max.V );
                pointCorner = face.Evaluate( corner );

                break;
              }
            }
          }
        }
        // Create the curves that will be used for the part division.
        IList<Curve> curveList = new List<Curve>();
        Curve curve1 = app.Create.NewLine( pointCenter, pointRight, true );
        curveList.Add( curve1 );
        Curve curve2 = app.Create.NewLine( pointRight, pointCorner, true );
        curveList.Add( curve2 );
        Curve curve3 = app.Create.NewLine( pointCorner, pointTop, true );
        curveList.Add( curve3 );
        Curve curve4 = app.Create.NewLine( pointTop, pointCenter, true );
        curveList.Add( curve4 );

        // intersectingReferenceIds will be empty for this example.
        ICollection<ElementId> intersectingReferenceIds = new List<ElementId>();

        // Divide the part
        Transaction dividePartTransaction = new Transaction( doc, "Divide Part" );
        dividePartTransaction.Start();
        PartMaker maker = PartUtils.DivideParts( doc, elementIdsToDivide, intersectingReferenceIds, curveList, sketchPlane.Id );
        dividePartTransaction.Commit();
        ICollection<ElementId> divElems = maker.GetDividedElementIds(); // Get the ids of the divided elements

        // Set the view's "Parts Visibility" parameter so that parts are shown
        Parameter partVisInView = doc.ActiveView.get_Parameter( BuiltInParameter.VIEW_PARTS_VISIBILITY );
        Transaction setPartVizTransaction = new Transaction( doc, "Set View Parameter" );
        setPartVizTransaction.Start();
        partVisInView.Set( 0 ); // 0 = Show Parts, 1 = Show Original, 2 = Show Both
        setPartVizTransaction.Commit();

        return Result.Succeeded;
      }
    }

    void OffsetPartFaces( Part part, double offsetCtr )
    {
      Autodesk.Revit.DB.GeometryElement geomElem = part.get_Geometry( new Options() );
      foreach( GeometryObject geomObject in geomElem.Objects )
      {
        if( geomObject is Solid )
        {
          Solid solid = geomObject as Solid;
          FaceArray faceArray = solid.Faces;
          foreach( Face face in faceArray )
          {
            if( part.CanOffsetFace( face ) )
              part.SetFaceOffset( face, offsetCtr );
            offsetCtr += 0.1;
          }
        }
      }
    }

    public class CreateAssembliesAndViews : IExternalCommand
    {
      public Result Execute(
        ExternalCommandData commandData,
        ref string message,
        ElementSet elements )
      {
        Document doc = commandData.Application.ActiveUIDocument.Document;
        UIDocument uidoc = commandData.Application.ActiveUIDocument;

        ElementId categoryId = doc.get_Element( uidoc.Selection.GetElementIds().FirstOrDefault() ).Category.Id;

        if( AssemblyInstance.IsValidNamingCategory( doc, categoryId, uidoc.Selection.GetElementIds() ) )
        {
          Transaction transactionA = new Transaction( doc, "Create Assembly" );
          transactionA.Start();
          AssemblyInstance assemblyInstance = AssemblyInstance.Create( doc, uidoc.Selection.GetElementIds(), categoryId );
          transactionA.Commit(); // need to commit the transaction to complete the creation of the assembly instance so it can be accessed in the code below

          Transaction transactionB = new Transaction( doc, "Create Assembly Views" );
          transactionB.Start();
          assemblyInstance.AssemblyTypeName = "Assembly #1"; // rename the assembly
          if( assemblyInstance.AllowsAssemblyViewCreation() ) // check to see if views can be created for this assembly
          {
            ElementId titleblockId = doc.TitleBlocks.Cast<FamilySymbol>().First<FamilySymbol>().Id; // find a titleblock
            // create a sheet, 3d view, sections, a material takeoff, and parts list for this assembly
            ViewSheet viewSheet = AssemblyViewUtils.CreateSheet( doc, assemblyInstance.Id, titleblockId );
            View3D view3d = AssemblyViewUtils.Create3DOrthographic( doc, assemblyInstance.Id );
            ViewSection detailSectionA = AssemblyViewUtils.CreateDetailSection( doc, assemblyInstance.Id, AssemblyDetailViewOrientation.DetailSectionA );
            ViewSection detailSectionB = AssemblyViewUtils.CreateDetailSection( doc, assemblyInstance.Id, AssemblyDetailViewOrientation.DetailSectionB );
            ViewSection detailSectionH = AssemblyViewUtils.CreateDetailSection( doc, assemblyInstance.Id, AssemblyDetailViewOrientation.HorizontalDetail );
            View materialTakeoff = AssemblyViewUtils.CreateMaterialTakeoff( doc, assemblyInstance.Id );
            View partList = AssemblyViewUtils.CreatePartList( doc, assemblyInstance.Id );

            // Add graphical views to the newly created sheet. Schedules (the Parts List and Material Takeoff) cannot be added to sheets with the 2012 API.
            viewSheet.AddView( view3d, new UV( 1, 1 ) );
            viewSheet.AddView( detailSectionA, new UV( 1.3, 1 ) );
            viewSheet.AddView( detailSectionB, new UV( 1, 1.3 ) );
            viewSheet.AddView( detailSectionH, new UV( 1.3, 1.3 ) );
          }
          transactionB.Commit();
        }
        return Result.Succeeded;
      }
    }

    [Autodesk.Revit.Attributes.Transaction( Autodesk.Revit.Attributes.TransactionMode.Manual )]
    public class Create_Assembly_From_Group : IExternalCommand
    {
      static AddInId m_appId = new AddInId( new Guid( "CF184371-E65A-4a35-BB4F-2959CA87A42D" ) );
      public Result Execute( ExternalCommandData commandData, ref string message, ElementSet elements )
      {
        Document document = commandData.Application.ActiveUIDocument.Document;
        UIDocument uidocument = commandData.Application.ActiveUIDocument;
        Autodesk.Revit.ApplicationServices.Application app = commandData.Application.Application;

        Guid schemaGuid = new Guid( "720080CB-DA99-40DC-9415-E53F280AA2F0" );

        Transaction transaction = new Transaction( document );

        Autodesk.Revit.DB.Group group = document.GetElement( uidocument.Selection.PickObject( ObjectType.Element, "Select a group" ) ) as Autodesk.Revit.DB.Group;

        // does this group already have an associated assembly?
        Schema existingSchema = Schema.Lookup( schemaGuid );
        if( existingSchema != null )
        {
          Entity existingEntity = group.GetEntity( existingSchema );
          Field existingField = existingSchema.GetField( "relatedElementId" );
          if( existingField != null )
          {
            TaskDialog.Show( "Revit", "This group already has an associated assembly" );
            return Result.Cancelled;
          }
        }

        AssemblyInstance assemblyInstance = null;

        ICollection<ElementId> ids = new Collection<ElementId>();
        foreach( Element e in group.Members )
        {
          ids.Add( e.Id );
        }
        transaction.Start( "Create Assembly" );
        assemblyInstance = AssemblyInstance.Create( document, ids, group.Members.get_Item( 0 ).Category.Id );
        transaction.Commit();
        transaction.Start( "Name Assembly" );
        assemblyInstance.AssemblyTypeName = "Assembly from Group: " + group.Name;
        transaction.Commit();
        // in this GroupType, store the id of associated AssemblyTypeName


        SchemaBuilder schemaBuilder = new SchemaBuilder( schemaGuid );
        schemaBuilder.SetSchemaName( "mySchema" );
        string fieldName = "relatedElementId";
        FieldBuilder fieldBuilder = schemaBuilder.AddSimpleField( fieldName, typeof( ElementId ) );
        Schema schema = schemaBuilder.Finish(); // register the Schema object

        Entity entity = new Entity( schema ); // create an entity (object) for this schema (class)
        Field field = schema.GetField( fieldName ); // get the field from the schema

        transaction.Start( "Set Entity Data" );
        entity.Set<ElementId>( fieldName, assemblyInstance.Id );
        group.GroupType.SetEntity( entity ); // store the assembly id in the group
        transaction.Commit();

        transaction.Start( "Set Entity Data" );
        entity.Set<ElementId>( fieldName, group.GroupType.Id );
        assemblyInstance.SetEntity( entity ); // store the group type id in the assembly
        transaction.Commit();
        return Result.Succeeded;
      }
    }

    [Autodesk.Revit.Attributes.Transaction( Autodesk.Revit.Attributes.TransactionMode.Automatic )]
    public class Update_Assemblies_From_Groups : IExternalCommand
    {
      static AddInId m_appId = new AddInId( new Guid( "CF184361-E65A-4a35-BA4F-2959CA87A42D" ) );
      public Result Execute( ExternalCommandData commandData, ref string message, ElementSet elements )
      {
        Document document = commandData.Application.ActiveUIDocument.Document;
        UIDocument uidocument = commandData.Application.ActiveUIDocument;
        Autodesk.Revit.ApplicationServices.Application app = commandData.Application.Application;

        Schema schema = Schema.Lookup( new Guid( "720080CB-DA99-40DC-9415-E53F280AA2F0" ) );
        if( schema == null ) return Result.Cancelled;
        Field field = schema.GetField( "relatedElementId" );

        FilteredElementCollector assemblyCollector = new FilteredElementCollector( document );
        foreach( Element assemblyInstanceElement in assemblyCollector.OfClass( typeof( AssemblyInstance ) ).ToElements() )
        {
          AssemblyInstance assemblyInstance = assemblyInstanceElement as AssemblyInstance;
          Entity entity = assemblyInstanceElement.GetEntity( schema );
          ElementId groupId = entity.Get<ElementId>( field );
          if( groupId != null ) // this assembly instance has an associated group
          {
            GroupType groupType = document.get_Element( groupId ) as GroupType;
            if( groupType == null ) // this group does not exist
            {
              assemblyInstance.Disassemble();
            }
            else
            {
              GroupSet groupSet = groupType.Groups;
              if( groupSet.Size > 0 )
              {
                IEnumerator iEnum = groupSet.GetEnumerator();
                iEnum.MoveNext();
                Autodesk.Revit.DB.Group group = iEnum.Current as Autodesk.Revit.DB.Group;
                ICollection<ElementId> ids = new Collection<ElementId>();
                foreach( Element e in group.Members )
                {
                  ids.Add( e.Id );
                }
                assemblyInstance.SetMemberIds( ids );
              }
            }
          }
        }
        return Result.Succeeded;
      }
    }

    [Autodesk.Revit.Attributes.Transaction( Autodesk.Revit.Attributes.TransactionMode.Manual )]
    public class ReportStoredId : IExternalCommand
    {
      static AddInId m_appId = new AddInId( new Guid( "CF184371-E66A-4a35-BA4F-2959CA87A42D" ) );
      public Result Execute( ExternalCommandData commandData, ref string message, ElementSet elements )
      {
        Document doc = commandData.Application.ActiveUIDocument.Document;
        UIDocument uidocument = commandData.Application.ActiveUIDocument;
        Autodesk.Revit.ApplicationServices.Application app = commandData.Application.Application;
        Guid schemaGuid = new Guid( "720080CB-DA99-40DC-9415-E53F280AA2F0" );
        string fieldName = "relatedElementId";

        Element e = doc.GetElement( uidocument.Selection.PickObject( ObjectType.Element ) );
        Autodesk.Revit.DB.Group group = e as Autodesk.Revit.DB.Group;
        GroupType groupType = group.GroupType;
        Schema schema = Schema.Lookup( schemaGuid );
        Entity retrievedEntity = groupType.GetEntity( schema );
        ElementId assemblyId = retrievedEntity.Get<ElementId>( schema.GetField( fieldName ) );

        TaskDialog.Show( "Revit", "ID = " + assemblyId );
        return Result.Succeeded;
      }
    }
    #endregion // Harry Mattison AU 2011 CP5211 sample code
  }
}
