#region Namespaces
using System;
using System.Collections.Generic;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Mechanical;
using Autodesk.Revit.UI;
using System.Diagnostics;
#endregion

namespace DuctEndPoints
{
  [Transaction( TransactionMode.ReadOnly )]
  public class Command : IExternalCommand
  {
    /// <summary>
    /// Return an English plural suffix for the given 
    /// number of items, i.e. 's' for zero or more 
    /// than one, and nothing for exactly one.
    /// </summary>
    public static string PluralSuffix( int n )
    {
      return 1 == n ? "" : "s";
    }

    /// <summary>
    /// Return a string for a real number 
    /// formatted to two decimal places.
    /// </summary>
    public static string RealString( double a )
    {
      return a.ToString( "0.##" );
    }

    /// <summary>
    /// Return a string for an XYZ point
    /// or vector with its coordinates 
    /// formatted to two decimal places.
    /// </summary>
    public static string PointString( XYZ p )
    {
      return string.Format( "({0},{1},{2})",
        RealString( p.X ),
        RealString( p.Y ),
        RealString( p.Z ) );
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Document doc = uidoc.Document;

      FilteredElementCollector a 
        = new FilteredElementCollector( doc )
          .OfClass( typeof( Duct ) );

      int nDucts = 0;
      int nCurves = 0;

      foreach( Duct d in a )
      {
        ++nDucts;

        LocationCurve lc = d.Location as LocationCurve;

        Debug.Assert( null != lc, 
          "expected duct to have valid location curve" );

        if( null != lc )
        {
          ++nCurves;

          Curve c = lc.Curve;

          Debug.Print( "Duct {0} from {1} to {2}",
            d.Id.IntegerValue, 
            PointString( c.get_EndPoint( 0 ) ),
            PointString( c.get_EndPoint( 1 ) ) );
        }
      }
      Debug.Print(
        "{0} duct{1} analysed, and {2} curve{3} listed.",
        nDucts, PluralSuffix( nDucts ),
        nCurves, PluralSuffix( nCurves ) );

      return Result.Succeeded;
    }
  }
}
