#region Namespaces
using System;
using System.Collections.Generic;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

namespace EquiDistantPointEval
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    #region CurveSelectionFilter
    class CurveSelectionFilter : ISelectionFilter
    {
      public bool AllowElement( Element e )
      {
        //Autodesk.Revit.DB.DetailCurve
        //  Autodesk.Revit.DB.DetailArc
        //  Autodesk.Revit.DB.DetailEllipse
        //  Autodesk.Revit.DB.DetailLine
        //  Autodesk.Revit.DB.DetailNurbSpline

        //Autodesk.Revit.DB.ModelCurve
        //  Autodesk.Revit.DB.ModelArc
        //  Autodesk.Revit.DB.ModelEllipse
        //  Autodesk.Revit.DB.ModelHermiteSpline
        //  Autodesk.Revit.DB.ModelLine
        //  Autodesk.Revit.DB.ModelNurbSpline

        return e is DetailLine
          || e is ModelLine
          || e is DetailArc
          || e is ModelArc
          || e is DetailNurbSpline
          || e is ModelNurbSpline;
      }

      public bool AllowReference( Reference r, XYZ p )
      {
        return false;
      }
    }
    #endregion // CurveSelectionFilter

    /// <summary>
    /// Create a horizontal detail curve circle of 
    /// the given radius at the specified point.
    /// </summary>
    DetailArc CreateCircle(
      Document doc,
      XYZ location,
      double radius )
    {
      XYZ norm = XYZ.BasisZ;

      double startAngle = 0;
      double endAngle = 2 * Math.PI;

      Plane plane = new Plane( norm, location );

      Arc arc = Arc.Create( plane,
        radius, startAngle, endAngle );

      return doc.Create.NewDetailCurve(
        doc.ActiveView, arc ) as DetailArc;
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Document doc = uidoc.Document;

      Reference r = null;

      try
      {
        r = uidoc.Selection.PickObject(
          ObjectType.Element,
          new CurveSelectionFilter(),
          "Please pick an arc or spline to select path" );
      }
      catch( Autodesk.Revit.Exceptions
        .OperationCanceledException )
      {
        return Result.Cancelled;
      }

      if( null == r )
      {
        message = "Null pick object reference.";
        return Result.Failed;
      }

      Element e = doc.GetElement( r );

      if( null == e || !( e is CurveElement ) )
      {
        message = "Not a curve element.";
        return Result.Failed;
      }

      // Extract data from the selected curve.

      Curve curve = ( e as CurveElement ).GeometryCurve;

      IList<XYZ> tessellation = curve.Tessellate();

      // Create a list of equi-distant points.

      List<XYZ> pts = new List<XYZ>( 1 );

      double stepsize = 5.0;
      double dist = 0.0;

      XYZ p = curve.GetEndPoint( 0 );

      foreach( XYZ q in tessellation )
      {
        if( 0 == pts.Count )
        {
          pts.Add( p );
          dist = 0.0;
        }
        else
        {
          dist += p.DistanceTo( q );

          if( dist >= stepsize )
          {
            pts.Add( q );
            dist = 0;
          }
          p = q;
        }
      }

      // Place a marker circle at each point.

      using( Transaction tx = new Transaction( doc ) )
      {
        tx.Start( "Draw Curves at Points" );

        foreach( XYZ pt in pts )
        {
          CreateCircle( doc, pt, 1 );
        }
        tx.Commit();
      }
      return Result.Succeeded;
    }
  }
}
