using System;
using System.IO;
using System.Linq;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Events;
using Autodesk.Revit.DB.ExtensibleStorage;
using Autodesk.Revit.UI;
using Common;

namespace ExtStorageSecondStep
{
  public class MoveExtensibleStorageDataApplication : IExternalApplication
  {
    private Guid? idToMove;

    public Result OnStartup( 
      UIControlledApplication application )
    {
      application.ControlledApplication.FamilyLoadingIntoDocument += OnFamilyLoadingIntoDocument;
      application.ControlledApplication.FamilyLoadedIntoDocument += OnFamilyLoadedIntoDocument;
      return Result.Succeeded;
    }

    public Result OnShutdown( 
      UIControlledApplication application )
    {
      application.ControlledApplication.FamilyLoadingIntoDocument -= OnFamilyLoadingIntoDocument;
      application.ControlledApplication.FamilyLoadedIntoDocument -= OnFamilyLoadedIntoDocument;
      return Result.Succeeded;
    }

    private static string GetFamilyFileName(
      FamilyLoadingIntoDocumentEventArgs e )
    {
      return string.Format( "{0}.rfa", e.FamilyName );
    }

    public static Document GetFamilyDocument(
      FamilyLoadingIntoDocumentEventArgs e )
    {
      var document = e.Document;
      var application = document.Application;

      var documents = application.Documents
        .Cast<Document>();

      string familyname = GetFamilyFileName( e );

      // If e.FamilyPath is not empty it is simple: 
      // just retrieve from the documents collection 
      // by document PathName;
      // otherwise, get the document by title 
      // (it can be with or without *.rfa).

      return string.IsNullOrWhiteSpace( e.FamilyPath )

        ? documents.FirstOrDefault(
          x =>
            string.IsNullOrWhiteSpace( x.PathName )
            && ( x.Title == e.FamilyName
              || x.Title == familyname ) )

        : documents.FirstOrDefault(
          x =>
            x.PathName == Path.Combine(
              e.FamilyPath, familyname ) );
    }

    private void OnFamilyLoadingIntoDocument(
      object sender,
      FamilyLoadingIntoDocumentEventArgs e )
    {
      var familyDocument = GetFamilyDocument( e );
      var family = familyDocument.OwnerFamily;

      var schema = SchemaHelper.GetFamilySchema();
      var entity = family.GetEntity( schema );

      // If the entity is valid, store it in 
      // the idToMove class property.

      idToMove = entity.IsValid()
        ? entity.Get<Guid>( "Id" )
        : (Guid?) null;
    }

    private void OnFamilyLoadedIntoDocument(
      object sender,
      FamilyLoadedIntoDocumentEventArgs e )
    {
      // The event can be cancelled or loading family 
      // can be without extensible storage entity.

      if( e.IsCancelled()
        || e.Status != RevitAPIEventStatus.Succeeded
        || !idToMove.HasValue )
      {
        return;
      }

      using( var transaction = new SubTransaction( e.Document ) )
      {
        transaction.Start();

        var familyId =
          e.NewFamilyId == ElementId.InvalidElementId
            ? e.OriginalFamilyId
            : e.NewFamilyId;

        var family = e.Document.GetElement( familyId );
        var schema = SchemaHelper.GetFamilySchema();
        var entity = new Entity( schema );
        entity.Set( "Id", idToMove.Value );
        family.SetEntity( entity );
        transaction.Commit();
      }
    }
  }
}
