#region Namespaces
using System;
using System.Collections.Generic;
using System.Linq; // in System.Core
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
#endregion

// This application creates the SAT file of Individual floor and
// wall elements. For each wall/floor element the rest of elements
// are made invisible in current view and the current view with single
// wall/floor element is exported to SAT file.
// The SAT files are saved in system defined temp folder.

// N.B: Do not save the Revit file after export as the document is 
// dirtied due to commiting transactions.


namespace ExportRoom
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    /// <summary>
    /// Scott Conover's function to get all model elements.
    /// </summary>
    IList<Element> GetAllModelElements( Document doc )
    {
      List<Element> elements = new List<Element>();

      FilteredElementCollector collector
        = new FilteredElementCollector( doc )
          .WhereElementIsNotElementType();

      foreach( Element e in collector )
      {
        if( null != e.Category
          && e.Category.HasMaterialQuantities )
        {
          elements.Add( e );
        }
      }
      return elements;
    }

    /// <summary>
    /// Export the given element to a SAT file.
    /// </summary>
    /// <param name="e">Element to export</param>
    /// <param name="allElements">All other elements to temporarily hide</param>
    /// <param name="filename_prefix">Filename prefix</param>
    /// <returns></returns>
    void CreateSatFileFor(
      Element e,
      IList<Element> allElements,
      string filename_prefix )
    {
      Document doc = e.Document;

      // Keep this element visible and 
      // hide all other model elements

      Transaction trans = new Transaction( doc );
      trans.Start( "Hide Elements" );

      // Create element set other than current wall/floor 

      ElementSet eset = new ElementSet();

      foreach( Element ele in allElements )
      {
        if( e.Id.IntegerValue != ele.Id.IntegerValue )
        {
          if( ele.CanBeHidden( doc.ActiveView ) )
            eset.Insert( ele );
        }
      }

      // Hide all elements other than current 
      // one in current view

      doc.ActiveView.Hide( eset );

      // Commit the transaction so that 
      // visibility is affected

      trans.Commit();

      // Export the ActiveView containing current 
      // element to SAT file

      SATExportOptions satExportOptions
        = new SATExportOptions();

      ViewSet vset = new ViewSet();
      vset.Insert( doc.ActiveView );

      // Get the material information

      IEnumerator<Material> mats =
        e.Materials.Cast<Material>().GetEnumerator();

      // Get the last material, as walls may 
      // have multiple materials assigned

      Material mat = null;

      while( mats.MoveNext() )
      {
        mat = mats.Current;
      }

      // Get temp folder path for saving SAT files

      string folder = System.IO.Path.GetTempPath();

      string filename = filename_prefix
        + "-" + e.Id.ToString()
        + "-" + mat.Name
        + "-" + mat.Id.ToString()
        + ".sat";

      doc.Export( folder, filename, vset,
        satExportOptions );

      // After export make all model elements visible

      trans = new Transaction( doc );
      trans.Start( "Unhide Elements" );
      doc.ActiveView.Unhide( eset );
      trans.Commit();
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Document doc = uidoc.Document;

      try
      {
        // Get all elements with material 
        // for hiding all except current

        IList<Element> allElements 
          = GetAllModelElements( doc );

        // Process all walls

        FilteredElementCollector col =
          new FilteredElementCollector( doc )
            .WhereElementIsNotElementType()
            .OfCategory( BuiltInCategory.OST_Walls );

        foreach( Wall wall in col )
        {
          CreateSatFileFor( wall, allElements, "wall" );
        }

        // Process all floors

        col = new FilteredElementCollector( doc )
          .WhereElementIsNotElementType()
          .OfCategory( BuiltInCategory.OST_Floors );

        foreach( Floor floor in col )
        {
          CreateSatFileFor( floor, allElements, "floor" );
        }
        return Result.Succeeded;
      }
      catch
      {
        TaskDialog.Show( "Error", "Could not create SAT Files" );
        return Result.Failed;
      }
    }
  }
}
