#region Namespaces
using System;
using System.Collections.Generic;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using AcRvUI=Autodesk.Revit.UI;
using Autodesk.Revit.UI;
using System.Linq; // in System.Core 
using System.IO; 
#endregion


// This application creates the SAT file of Individual floor and
// wall elements. For each wall/floor element the rest of elements
// are made invisible in current view and the current view with single
// wall/floor element is exported to SAT file.
// The SAT files are saved in system defined temp folder.

// N.B: Do not save the Revit file after export as the document is 
// dirtied due to commiting transactions.


namespace ExportRoom
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    Application _rvtApp;
    Document _rvtDoc;


    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      _rvtApp = uiapp.Application;
      _rvtDoc = uidoc.Document;

      {
        try
        {

          // Get all walls

          IList<Element> walls =
            findElements( BuiltInCategory.OST_Walls );

          // Get all floors

          IList<Element> floors =
            findElements( BuiltInCategory.OST_Floors );

          // Get all elements with material
          IList<Element> allElements = getAllModelElements();

          createSATFiles( walls,
                          allElements,
            /* filename prefix */ "wall-"
                        );
          createSATFiles( floors,
                          allElements,
            /* filename prefix */ "floor-" );


        }
        catch
        {
          TaskDialog.Show( "Error", "Could not create SAT Files" );
          return Result.Failed;
        }
      }

      return Result.Succeeded;
    }

    // Function to create SAT files
    Result createSATFiles( IList<Element> elements,
                          IList<Element> allElements,
                          string stag
                         )
    {

      foreach( Element roomele in elements )
      {
        // Keep this wall/floor on and put off rest model elements

        Transaction trans = new Transaction( _rvtDoc );
        trans.Start( "HideUnHideWall" );

        // Create element set other than current wall/floor 

        ElementSet eset = new ElementSet();
        foreach( Element ele in allElements )
        {
          if( roomele.Id.IntegerValue != ele.Id.IntegerValue )
          {
            if( ele.CanBeHidden( _rvtDoc.ActiveView ) )
              eset.Insert( ele );

          }

        }

        // Hide all elements other than current wall/floor 
        // in current view

        _rvtDoc.ActiveView.Hide( eset );

        // Commit the transaction so that visibility is affected

        trans.Commit();

        // Export the ActiveView which contians current wall/floor 
        // to SAT file

        SATExportOptions satExportOptions = new SATExportOptions();
        ViewSet vset = new ViewSet();
        vset.Insert( _rvtDoc.ActiveView );

        // Get the material information

        IEnumerator<Material> mats =
          roomele.Materials.Cast<Material>().GetEnumerator();

        // Get the last material as walls may have multiple 
        // materials assigned

        Material mat = null;
        while( mats.MoveNext() )
        {
          mat = mats.Current;
        }


        // Get temp folder path for saving SAT files
        string temp = System.IO.Path.GetTempPath();

        _rvtDoc.Export(/* folder */ temp,
          /* filename */ stag + roomele.Id.ToString() +
                        "-" + mat.Name + "-" + mat.Id.ToString() +
                        ".sat",
          /* ViewSet */ vset,
                        satExportOptions
                      );

        // After export make all model elements visible

        trans = new Transaction( _rvtDoc );
        trans.Start( "temp" );
        _rvtDoc.ActiveView.Unhide( eset );
        trans.Commit();


      }

      return Result.Succeeded;
    }


    #region Helper Functions


    IList<Element> findElements( BuiltInCategory builCat )
    {
      // Find all Wall instances in the document by using category 
      // filter

      ElementCategoryFilter filter =
        new ElementCategoryFilter( builCat );

      // Apply the filter to the elements in the active document,
      // Use shortcut WhereElementIsNotElementType() to find 
      // specific instances only

      FilteredElementCollector collector =
        new FilteredElementCollector( _rvtDoc );

      IList<Element> catElements =
        collector.WherePasses( filter ).WhereElementIsNotElementType()
        .ToElements();

      if( catElements.Count > 0 )
      {
        return catElements;
      }
      return null;
    }


    // Scott Conover's function to get all model elements
    IList<Element> getAllModelElements() // get all elements
    {
      List<Element> elements = new List<Element>();

      FilteredElementCollector collector
        = new FilteredElementCollector( _rvtDoc )
          .WhereElementIsNotElementType();

      foreach( Element e in collector )
      {
        if( null != e.Category && e.Category.HasMaterialQuantities )
        {
          elements.Add( e );
        }
      }
      return elements;
    }

    #endregion // Helper Functions
  }
}
