#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Events;
using Autodesk.Revit.UI.Selection;
#endregion

namespace ExportWallboard
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    #region WallPartSelectionFilter
    class WallPartSelectionFilter : ISelectionFilter
    {
      public bool AllowElement( Element e )
      {
        bool rc = false;

        Part part = e as Part;

        if( null != part )
        {
          ICollection<ElementId> cids
            = part.GetSourceElementOriginalCategoryIds();

          if( 1 == cids.Count )
          {
            ElementId cid = cids.First<ElementId>();

            BuiltInCategory bic
              = (BuiltInCategory) cid.IntegerValue;

            rc = ( BuiltInCategory.OST_Walls == bic );
          }
        }
        return rc;
      }

      public bool AllowReference( Reference r, XYZ p )
      {
        return true;
      }
    }
    #endregion // WallPartSelectionFilter

    const string _caption = "Export Wallboard";
    const string _folder = "Y:/a/src/revit/export_wallboard/test";

    static void InfoMsg( string msg )
    {
      Debug.Print( msg );
      TaskDialog.Show( _caption, msg );
    }

    static void ErrorMsg( string msg )
    {
      Debug.Print( msg );
      TaskDialog dlg = new TaskDialog( _caption );
      dlg.MainIcon = TaskDialogIcon.TaskDialogIconWarning;
      dlg.MainInstruction = msg;
      dlg.Show();
    }

    void OnDialogBoxShowing(
      object sender,
      DialogBoxShowingEventArgs e )
    {
      TaskDialogShowingEventArgs e2
        = e as TaskDialogShowingEventArgs;

      if( null != e2 && e2.DialogId.Equals(
        "TaskDialog_Really_Print_Or_Export_Temp_View_Modes" ) )
      {
        int cmdLink
          = (int) TaskDialogResult.CommandLink2;

        e.OverrideResult( cmdLink );
      }
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      if( null == doc )
      {
        ErrorMsg( "Please run this command in a valid"
          + " Revit project document." );
        return Result.Failed;
      }

      View view = doc.ActiveView;

      if( null == view || !( view is View3D ) )
      {
        ErrorMsg( "Please run this command in a valid"
          + " 3D view." );
        return Result.Failed;
      }

      if( PartsVisibility.ShowPartsOnly
        != view.PartsVisibility )
      {
        ErrorMsg( "Please run this command in a view"
          + " displaying parts and not source elements." );
        return Result.Failed;
      }

      // Define the list of views to export, 
      // including only the current 3D view

      List<ElementId> viewIds = new List<ElementId>( 1 );
      viewIds.Add( view.Id );

      // Iterate over all pre-selected parts

      List<ElementId> ids = null;

      Selection sel = uidoc.Selection;

      if( 0 < sel.Elements.Size )
      {
        foreach( Element e in sel.Elements )
        {
          if( !( e is Part ) )
          {
            ErrorMsg( "Please pre-select only gyp wallboard"
              + " parts before running this command." );
            return Result.Failed;
          }

          Part part = e as Part;

          ICollection<LinkElementId> lids
            = part.GetSourceElementIds();

          if( 1 != lids.Count )
          {
            ErrorMsg( "Gyp wallboard parts has multiple"
              + " source elements." );
            return Result.Failed;
          }

          LinkElementId lid = lids.First<LinkElementId>();
          ElementId hostId = lid.HostElementId;
          ElementId linkedId = lid.LinkedElementId;
          ElementId parentId = hostId;
          ElementId partId = e.Id;

          // Determine parent category

          Element parent = doc.GetElement( parentId );
          Category cat = parent.Category;

          ICollection<ElementId> cids
            = part.GetSourceElementOriginalCategoryIds();

          if( 1 != cids.Count )
          {
            ErrorMsg( "Gyp wallboard parts has multiple"
              + " source element categories." );
            return Result.Failed;
          }

          ElementId cid = cids.First<ElementId>();

          //cat = doc.GetElement( id ) as Category;

          // Expected parent category is OST_Walls

          BuiltInCategory bic
            = (BuiltInCategory) cid.IntegerValue;

          if( BuiltInCategory.OST_Walls != bic )
          {
            ErrorMsg( "Pleqase pre-select only "
              + " gyp wallboard parts." );
            return Result.Failed;
          }

          if( null == ids )
          {
            ids = new List<ElementId>( 1 );
          }

          ids.Add( partId );
        }

        if( null == ids )
        {
          ErrorMsg( "Please pre-select only gyp wallboard"
            + " parts before running this command." );
          return Result.Failed;
        }
      }

      // If no parts were pre-selected, 
      // prompt for post-selection

      if( null == ids )
      {
        IList<Reference> refs = null;

        try
        {
          refs = sel.PickObjects( ObjectType.Element,
            new WallPartSelectionFilter(),
            "Please select wall parts." );
        }
        catch( Autodesk.Revit.Exceptions
          .OperationCanceledException )
        {
          return Result.Cancelled;
        }
        ids = new List<ElementId>(
          refs.Select<Reference, ElementId>(
            r => r.ElementId ) );
      }

      try
      {
        // Register event handler for 
        // "TaskDialog_Really_Print_Or_Export_Temp_View_Modes" 
        // dialogue

        uiapp.DialogBoxShowing
          += new EventHandler<DialogBoxShowingEventArgs>(
            OnDialogBoxShowing );

        //DXFExportOptions predefinedOptions 
        //  = DXFExportOptions.GetPredefinedOptions( 
        //    doc, ??? );

        DXFExportOptions opt = new DXFExportOptions();

        //opt.FileVersion = ACADVersion.R2000;

        string filename;

        using( TransactionGroup txg = new TransactionGroup( doc ) )
        {
          txg.Start( "Export Wall Parts" );

          foreach( ElementId id in ids )
          {
            Element e = doc.GetElement( id );

            Debug.Assert( e is Part,
              "expected parts only" );

            Part part = e as Part;

            ICollection<LinkElementId> lids
              = part.GetSourceElementIds();

            Debug.Assert( 1 == lids.Count,
              "unexpected multiple part source elements." );

            LinkElementId lid = lids.First<LinkElementId>();
            ElementId hostId = lid.HostElementId;
            ElementId linkedId = lid.LinkedElementId;
            ElementId parentId = hostId;
            ElementId partId = e.Id;

            filename = string.Format( "{0}_{1}",
              parentId, partId );

            if( view.IsTemporaryHideIsolateActive() )
            {
              using( Transaction tx = new Transaction( doc ) )
              {
                tx.Start( "Disable Temporary Isolate" );

                view.DisableTemporaryViewMode(
                  TemporaryViewMode.TemporaryHideIsolate );

                tx.Commit();
              }

              Debug.Assert( !view.IsTemporaryHideIsolateActive(),
                "expected to turn off temporary hide/isolate" );
            }

            using( Transaction tx = new Transaction( doc ) )
            {
              tx.Start( "Export Wall Part "
                + partId.ToString() );

              view.IsolateElementTemporary( partId ); // requires transaction

              //List<ElementId> unhideIds = new List<ElementId>( 1 );
              //unhideIds.Add( partId );
              //view.UnhideElements( unhideIds );

              //doc.Regenerate(); // this is insufficient

              tx.Commit();
            }

            doc.Export( _folder, filename, viewIds, opt );

            // We do not commit the transaction group, 
            // because no modifications should be saved.
            // The transaction group is only created and 
            // started to encapsulate the transactions 
            // required by the IsolateElementTemporary 
            // method. Since the transaction group is not 
            // committed, the changes are automatically 
            // discarded.

            //txg.Commit();
          }
        }
      }
      finally
      {
        uiapp.DialogBoxShowing
          -= new EventHandler<DialogBoxShowingEventArgs>(
            OnDialogBoxShowing );
      }
      return Result.Succeeded;
    }
  }
}
