﻿#region Header
//
// (C) Copyright 2011 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE. AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
#endregion // Header

#region Namespaces
using System;
using System.Diagnostics;
using System.Collections.Generic;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.ExtensibleStorage;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion // Namespaces

namespace ExtensibleStorage
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    static string PluralSuffix( int n )
    {
      return 1 == n ? "" : "s";
    }

    /// <summary>
    /// Create an extensible storage schema, 
    /// attach it to a wall, populate it with data, 
    /// and retrieve the data back from the wall.
    /// </summary>
    void StoreDataInWall(
      Wall wall,
      XYZ dataToStore )
    {
      SchemaBuilder schemaBuilder = new SchemaBuilder(
        new Guid( "720080CB-DA99-40DC-9415-E53F280AA1F0" ) );

      // allow anyone to read the object

      schemaBuilder.SetReadAccessLevel(
        AccessLevel.Public );

      // restrict writing to this vendor only

      schemaBuilder.SetWriteAccessLevel(
        AccessLevel.Vendor );

      // required because of restricted write-access

      schemaBuilder.SetVendorId( "ADNP" );

      // create a field to store an XYZ

      FieldBuilder fieldBuilder = schemaBuilder
        .AddSimpleField( "WireSpliceLocation",
        typeof( XYZ ) );

      fieldBuilder.SetUnitType( UnitType.UT_Length );

      fieldBuilder.SetDocumentation( "A stored "
        + "location value representing a wiring "
        + "splice in a wall." );

      schemaBuilder.SetSchemaName( "WireSpliceLocation" );

      // register the schema

      Schema schema = schemaBuilder.Finish(); 

      // create an entity (object) for this schema (class)

      Entity entity = new Entity( schema );

      // get the field from the schema

      Field fieldSpliceLocation = schema.GetField(
        "WireSpliceLocation" );

      // set the value for this entity

      entity.Set<XYZ>( fieldSpliceLocation, dataToStore,
        DisplayUnitType.DUT_METERS );

      // store the entity on the element

      wall.SetEntity( entity );

      // read back the data from the wall

      Entity retrievedEntity = wall.GetEntity( schema );

      XYZ retrievedData = retrievedEntity.Get<XYZ>(
        schema.GetField( "WireSpliceLocation" ),
        DisplayUnitType.DUT_METERS );
    }

    /// <summary>
    /// Create an extensible storage schema specifying 
    /// a dictionary mapping keys to values, both using 
    /// strings,  populate it with data, attach it to the 
    /// given element, and retrieve the data back again.
    /// </summary>
    void StoreStringMapInElement( Element e )
    {
      SchemaBuilder schemaBuilder = new SchemaBuilder(
        new Guid( "F1697E22-9338-4A5C-8317-5B6EE088ECB4" ) );

      // allow anyone to read or write the object

      schemaBuilder.SetReadAccessLevel(
        AccessLevel.Public );

      schemaBuilder.SetWriteAccessLevel(
        AccessLevel.Public );

      // create a field to store a string map

      FieldBuilder fieldBuilder
        = schemaBuilder.AddMapField( "StringMap",
          typeof( string ), typeof( string ) );

      fieldBuilder.SetDocumentation(
        "A string map for Tobias." );

      schemaBuilder.SetSchemaName(
        "TobiasStringMap" );

      // register the schema

      Schema schema = schemaBuilder.Finish();

      // create an entity (object) for this schema (class)

      Entity entity = new Entity( schema );

      // get the field from the schema

      Field field = schema.GetField(
        "StringMap" );

      // set the value for this entity

      IDictionary<string, string> stringMap
        = new Dictionary<string, string>();

      stringMap.Add( "key1", "value1" );
      stringMap.Add( "key2", "value2" );

      entity.Set<IDictionary<string, string>>(
        field, stringMap );

      // store the entity on the element

      e.SetEntity( entity );

      // read back the data from the wall

      Entity retrievedEntity = e.GetEntity( schema );

      IDictionary<string, string> stringMap2
        = retrievedEntity
        .Get<IDictionary<string, string>>(
          schema.GetField( "StringMap" ) );
    }

    /// <summary>
    /// List all schemas in Revit memory across all documents.
    /// </summary>
    void ListSchemas()
    {
      IList<Schema> schemas = Schema.ListSchemas();

      int n = schemas.Count;

      Debug.Print( 
        string.Format( "{0} schema{1} defined:", 
          n, PluralSuffix( n ) ) );

      foreach( Schema s in schemas )
      {
        IList<Field> fields = s.ListFields();

        n = fields.Count;

        Debug.Print( 
          string.Format( "Schema '{0}' has {1} field{2}:", 
            s.SchemaName, n, PluralSuffix( n ) ) );

        foreach( Field f in fields )
        {
          Debug.Print(
            string.Format( 
              "Field '{0}' has value type {1}"
              + " and unit type {2}", f.FieldName, 
              f.ValueType, f.UnitType ) );
        }
      }
    }

    class WallFilter : ISelectionFilter
    {
      public bool AllowElement( Element e )
      {
        return e is Wall;
      }

      public bool AllowReference( Reference r, XYZ p )
      {
        return true;
      }
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Document doc = uidoc.Document;

      try
      {
        // pick an element and define the XYZ 
        // data to store at the same time

        Reference r = uidoc.Selection.PickObject( 
          ObjectType.Face, 
          new WallFilter(), 
          "Please pick a wall at a point on one of its faces" );

        Wall wall = doc.get_Element( r.ElementId ) as Wall;
        XYZ dataToStore = r.GlobalPoint;

        Transaction t = new Transaction( doc,
          "Create Extensible Storage Schemata and Store Data" ); 

        t.Start();

        // store the data, and also 
        // demonstrate reading it back

        StoreDataInWall( wall, dataToStore );

        StoreStringMapInElement( wall );

        t.Commit();

        // list all schemas in memory across all documents

        ListSchemas();

        return Result.Succeeded;
      }
      catch( Exception ex )
      {
        message = ex.Message;
        return Result.Failed;
      }
    }
  }
}

// if the vendor id of the add-in manifest does not match the schema vendor 
// id, an exception is thrown by the call to wall.SetEntity( entity ):
// Writing of Entities of this Schema is not allowed to the current add-in. 
// A transaction or sub-transaction was opened but not closed. All changes 
// to the active document made by External Command will be discarded.
