﻿#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Events;
using Autodesk.Revit.UI;
#endregion

namespace FamilyApi
{
  /// <summary>
  /// Load table family if not already present and
  /// place table family instances.
  /// </summary>
  [Transaction( TransactionMode.Manual )]
  public class CmdTableLoadPlace : IExternalCommand
  {
    /// <summary>
    /// Family name.
    /// </summary>
    public const string FamilyName = "family_api_table";

    /// <summary>
    /// Family file path.
    /// Normally, you would either search the  library
    /// paths provided by Application.GetLibraryPaths 
    /// method. In this case, we store the sample 
    /// family in the same location as the add-in.
    /// </summary>
    //const string _family_folder = "Z:/a/rvt";
    static string _family_folder
      = Path.GetDirectoryName(
        typeof( CmdTableLoadPlace )
          .Assembly.Location );

    /// <summary>
    /// Family filename extension RFA.
    /// </summary>
    const string _family_ext = "rfa";

    /// <summary>
    /// Family file path
    /// </summary>
    static string _family_path = null;

    /// <summary>
    /// Return complete family file path
    /// </summary>
    static string FamilyPath
    {
      get
      {
        if( null == _family_path )
        {
          _family_path = Path.Combine(
            _family_folder, FamilyName );

          _family_path = Path.ChangeExtension(
            _family_path, _family_ext );
        }
        return _family_path;
      }
    }

    /// <summary>
    /// Collection of newly added family instances
    /// </summary>
    List<ElementId> _added_element_ids
      = new List<ElementId>();

    /// <summary>
    /// External command mainline
    /// </summary>
    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      // Retrieve the family if it is already present:

      Family family = Util.FindElementByName(
        doc, typeof( Family ), FamilyName ) as Family;

      if( null == family )
      {
        // It is not present, so check for 
        // the file to load it from:

        if( !File.Exists( FamilyPath ) )
        {
          Util.ErrorMsg( string.Format(
            "Please ensure that the sample table "
            + "family file '{0}' exists in '{1}'.",
            FamilyName, _family_folder ) );

          return Result.Failed;
        }

        // Load family from file:

        using( Transaction tx = new Transaction( doc ) )
        {
          tx.Start( "Load Family" );
          doc.LoadFamily( FamilyPath, out family );
          tx.Commit();
        }
      }

      // Determine the family symbol

      FamilySymbol symbol = null;

      foreach( FamilySymbol s in family.Symbols )
      {
        symbol = s;

        // Our family only contains one
        // symbol, so pick it and leave

        break;
      }

      // Place the family symbol:

      // Subscribe to document changed event to
      // retrieve family instance elements added by the 
      // PromptForFamilyInstancePlacement operation:

      app.DocumentChanged
        += new EventHandler<DocumentChangedEventArgs>(
          OnDocumentChanged );

      _added_element_ids.Clear();

      // PromptForFamilyInstancePlacement cannot 
      // be called inside transaction.

      uidoc.PromptForFamilyInstancePlacement( symbol );

      app.DocumentChanged
        -= new EventHandler<DocumentChangedEventArgs>(
          OnDocumentChanged );

      // Access the newly placed family instances:

      int n = _added_element_ids.Count();

      string msg = string.Format(
        "Placed {0} {1} family instance{2}{3}",
        n, family.Name,
        Util.PluralSuffix( n ),
        Util.DotOrColon( n ) );

      string ids = string.Join( ", ",
        _added_element_ids.Select<ElementId, string>(
          id => id.IntegerValue.ToString() ) );

      Util.InfoMsg2( msg, ids );

      return Result.Succeeded;
    }

    void OnDocumentChanged(
      object sender,
      DocumentChangedEventArgs e )
    {
      _added_element_ids.AddRange(
        e.GetAddedElementIds() );
    }
  }
}

// Create and load family, select and place family instance
// http://thebuildingcoder.typepad.com/blog/2011/06/creating-and-inserting-an-extrusion-family.html

// PromptForFamilyInstancePlacement
// http://thebuildingcoder.typepad.com/blog/2010/06/place-family-instance.html
