﻿#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

namespace FamilyApi
{
  /// <summary>
  /// Create a new table type and 
  /// modify selected instances.
  /// </summary>
  [Transaction( TransactionMode.Manual )]
  public class CmdTableNewTypeModify : IExternalCommand
  {
    const string _usage_error = "Please pre-select "
      + "only table elements before running "
      + "this command.";

    /// <summary>
    /// Name of the new table family type.
    /// </summary>
    const string _type_name = "1200x750x380mm";

    //#region TableSelectionFilter
    //class TableSelectionFilter : ISelectionFilter
    //{
    //  static public bool IsTable( Element e )
    //  {
    //    bool rc = false;

    //    Category cat = e.Category;

    //    if( null != cat )
    //    {
    //      if( cat.Id.IntegerValue.Equals(
    //        (int) BuiltInCategory.OST_Furniture ) )
    //      {
    //        FamilyInstance fi = e as FamilyInstance;

    //        if( null != fi )
    //        {
    //          string fname = fi.Symbol.Family.Name;

    //          rc = fname.Equals(
    //            CmdTableLoadPlace.FamilyName );
    //        }
    //      }
    //    }
    //    return rc;
    //  }

    //  public bool AllowElement( Element e )
    //  {
    //    return IsTable( e );
    //  }

    //  public bool AllowReference( Reference r, XYZ p )
    //  {
    //    return true;
    //  }
    //}
    //#endregion // TableSelectionFilter

    /// <summary>
    /// Set the value of the named parameter 
    /// on the given element to a new length
    /// specified in millimetres.
    /// </summary>
    void SetElementParameterInMm(
      Element e,
      string parameter_name,
      double lengthInMm )
    {
      e.get_Parameter( parameter_name )
        .Set( Util.MmToFoot( lengthInMm ) );
    }

    /// <summary>
    /// Create new table type
    /// Type Name: 1200x750x380mm
    /// Parameters to change:
    /// Width = 1200
    /// Depth = 750
    /// Height = 380
    /// Top Material = Glass
    /// Leg Material = Glass
    /// </summary>
    FamilySymbol CreateNewType( FamilySymbol oldType )
    {
      FamilySymbol sym = oldType.Duplicate(
        _type_name ) as FamilySymbol;

      SetElementParameterInMm( sym, "Width", 1200 );
      SetElementParameterInMm( sym, "Depth", 750 );
      SetElementParameterInMm( sym, "Height", 380 );

      Element material_glass = Util.FindElementByName(
        sym.Document, typeof( Material ), "Glass" );

      ElementId id = material_glass.Id;

      sym.get_Parameter( "Top Material" ).Set( id );
      sym.get_Parameter( "Leg Material" ).Set( id );

      return sym;
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      TableSelector selector
        = new TableSelector( uidoc );

      IList<FamilyInstance> tables
        = selector.SelectedTables;

      if( null == tables )
      {
        return selector.Return();
      }

      using( Transaction tx = new Transaction( doc ) )
      {
        tx.Start( "Create Type and Modify Instances" );

        // Retrieve existing type or create new

        FamilySymbol symbol
          = Util.FindElementByName( doc,
            typeof( FamilySymbol ), _type_name )
              as FamilySymbol
          ?? CreateNewType( tables[0].Symbol );

        foreach( FamilyInstance table in tables )
        {
          Debug.Print( Util.ElementDescription(
            table ) );

          table.Symbol = symbol;
        }

        tx.Commit();
      }
      return Result.Succeeded;
    }
  }
}
