﻿#region Namespaces
using System;
using System.Collections.Generic;
using System.Linq;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using System.Diagnostics;
#endregion

namespace FamilyApi
{
  class TableSelector
  {
    const string _usage_error = "Please pre-select "
      + "only table family instances to modify "
      + "before launching this command.";

    #region Common filtering helper method
    /// <summary>
    /// Determine whether the given element is a valid 
    /// table family instance, reused by both selection
    /// filter and pre-selection checking. This specific
    /// implementation requires a family instance element
    /// of the furniture category belonging to the named
    /// family.
    /// </summary>
    static public bool IsTable( Element e )
    {
      bool rc = false;

      Category cat = e.Category;

      if( null != cat )
      {
        if( cat.Id.IntegerValue.Equals(
          (int) BuiltInCategory.OST_Furniture ) )
        {
          FamilyInstance fi = e as FamilyInstance;

          if( null != fi )
          {
            string fname = fi.Symbol.Family.Name;

            rc = fname.Equals(
              CmdTableLoadPlace.FamilyName );
          }
        }
      }
      return rc;
    }
    #endregion // Common filtering helper method

    #region TableSelectionFilter
    class TableSelectionFilter : ISelectionFilter
    {
      public bool AllowElement( Element e )
      {
        return IsTable( e );
      }

      public bool AllowReference( Reference r, XYZ p )
      {
        return true;
      }
    }
    #endregion // TableSelectionFilter

    List<FamilyInstance> _tables;
    string _msg;
    Result _result;

    /// <summary>
    /// Return selected tables or null
    /// </summary>
    public IList<FamilyInstance> SelectedTables
    {
      get
      {
        return _tables;
      }
    }

    // There is no need to provide external access to 
    // the error message or result code, since that 
    // can all be encapsulated in the call to Return.

    /// <summary>
    /// Return error message in case
    /// of failure or cancellation
    /// </summary>
    //public string ErrorMessage
    //{
    //  get
    //  {
    //    return _msg;
    //  }
    //}

    /// <summary>
    /// Return selection result
    /// </summary>
    //public Result Result
    //{
    //  get
    //  {
    //    return _result;
    //  }
    //}

    /// <summary>
    /// Instantiate and run table selector
    /// </summary>
    public TableSelector( UIDocument uidoc )
    {
      _tables = null;
      _msg = null;

      Document doc = uidoc.Document;

      if( null == doc )
      {
        _msg = "Please run this command in a valid"
          + " Revit project document.";
        _result = Result.Failed;
      }

      // Check for a pre-selected tables

      Selection sel = uidoc.Selection;

      int n = sel.Elements.Size;

      if( 0 < n )
      {
        if( 1 != n )
        {
          _msg = _usage_error;
          _result = Result.Failed;
        }

        foreach( Element e in sel.Elements )
        {
          if( !IsTable( e ) )
          {
            _msg = _usage_error;
            _result = Result.Failed;
          }
          
          if( null == _tables )
          {
            _tables = new List<FamilyInstance>( n );
          }

          _tables.Add( e as FamilyInstance );
        }
      }

      // If no tables were pre-selected, 
      // prompt for post-selection

      if( null == _tables
        || 0 == _tables.Count )
      {
        IList<Reference> refs = null;

        try
        {
          refs = sel.PickObjects( ObjectType.Element,
            new TableSelectionFilter(),
            "Please select tables to be modified." );
        }
        catch( Autodesk.Revit.Exceptions
          .OperationCanceledException )
        {
          _result = Result.Cancelled;
        }

        if( null != refs && 0 < refs.Count )
        {
          _tables = new List<FamilyInstance>(
            refs.Select<Reference, FamilyInstance>(
              r => doc.GetElement( r.ElementId )
                as FamilyInstance ) );
        }
      }

      Debug.Assert(
        null == _tables || 0 < _tables.Count,
        "ensure that we do not return a non-null but empty collection" );

      _result = (null == _tables) // || 0 == _tables.Count
        ? Result.Cancelled 
        : Result.Succeeded;
    }

    /// <summary>
    /// Return the cancellation or failure code
    /// to Revit and display a message if there is 
    /// anything to say.
    /// </summary>
    public Result Return()
    {
      if( Result.Failed == _result )
      {
        Debug.Assert( 0 < _msg.Length,
          "expected error message" );

        Util.ErrorMsg( _msg );
      }
      return _result;
    }
  }
}
