#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using Autodesk.Revit.DB.Mechanical;
#endregion

namespace FlowParam
{
  [Transaction( TransactionMode.Manual )]
  public class Command : IExternalCommand
  {
    static Guid _shared_param_duct_max_airflow 
      = new Guid( "87b12ca4-8a4c-4731-bf88-f50bccd9c5d4" );

    /// <summary>
    /// Return the given element's connector manager, 
    /// using either the family instance MEPModel or 
    /// directly from the duct connector manager.
    /// </summary>
    static ConnectorManager GetConnectorManager( 
      Element e )
    {
      Duct duct = e as Duct;

      return null == duct
        ? ( e as FamilyInstance ).MEPModel.ConnectorManager
        : duct.ConnectorManager;
    }

    /// <summary>
    /// Retrieve max flow from all the given connectors.
    /// </summary>
    static double GetMaxFlow(
      ConnectorSet connectors )
    {
      double flow = 0.0;

      foreach( Connector c in connectors )
      {
        // Accessing flow property requires these
        // domains or throws an exception saying 
        // "Flow is available only for connectors 
        // of DomainHavc and DomainPiping."

        Domain d = c.Domain;

        if( Domain.DomainHvac != d
          && Domain.DomainPiping != d )
        {
          continue;
        }

        if( flow < c.Flow )
        {
          flow = c.Flow;
        }
      }
      return flow;
    }

    /// <summary>
    /// Set the max flow parameter on the given 
    /// element and return true on success.
    /// This is generic, so it can handle both
    /// ducts and fittings. Later, this proved
    /// unnecessary, and we use ot for ducts only.
    /// </summary>
    static bool SetMaxFlowOnElement( Element e )
    {
      ConnectorSet connectors
        = GetConnectorManager( e ).Connectors;

      int n = connectors.Size;

      double flow = GetMaxFlow( connectors );

      Debug.Print(
        "{0} has {1} connector{2} and max flow {3}.",
        Util.ElementDescription( e ), n, 
        Util.PluralSuffix( n ), flow );

      Parameter p = e.get_Parameter(
        _shared_param_duct_max_airflow );

      bool rc = false;

      if( null == p )
      {
        //Util.InfoMsg( "Please ensure that all "
        //  + "duct and their fittings have a "
        //  + "'Duct Max Airflow' shared parameter." );

        Debug.Print( "{0} has no 'Duct Max Airflow' "
          + "shared parameter.", 
          Util.ElementDescription( e ) );
      }
      else
      {
        // Store the max flow value in the specified
        // parameter on the given element.

        rc = p.Set( flow );
      }
      return rc;
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;

      if( null == uidoc )
      {
        Util.InfoMsg( "Please run this command "
          + "in a valid document context." );

        return Result.Failed;
      }

      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      int nDucts = 0;

      //int nFittings = 0;
      //int nFittingsFailed = 0;

      using( Transaction tx = new Transaction( doc ) )
      {
        tx.Start( Util.Caption );

        FilteredElementCollector ducts
          = new FilteredElementCollector( doc )
            .OfClass( typeof( Duct ) );

        foreach( Duct duct in ducts )
        {
          if( SetMaxFlowOnElement( duct ) )
          {
            ++nDucts;
          }
          else
          {
            return Result.Failed;
          }
        }

        #region Iterate over fittings
#if ITERATE_OVER_FITTINGS
        FilteredElementCollector fittings
          = new FilteredElementCollector( doc )
            .OfClass( typeof( FamilyInstance ) )
            .OfCategory( BuiltInCategory.OST_DuctFitting );

        foreach( FamilyInstance fitting in fittings )
        {
          if( SetMaxFlowOnElement( fitting ) )
          {
            ++nFittings;
          }
          else
          {
            // Return here to terminate if a fitting 
            // lacks the shared parameter.

            //return Result.Failed;

            ++nFittingsFailed;
          }
        }
#endif // ITERATE_OVER_FITTINGS
        #endregion // Iterate over fittings

        tx.Commit();
      }

      //Util.InfoMsg( string.Format(
      //  "Set max flow parameter on {0} duct{1} and {2} fitting{3}, {4} fitting{5} failed.",
      //  nDucts, Util.PluralSuffix( nDucts ),
      //  nFittings, Util.PluralSuffix( nFittings ),
      //  nFittingsFailed, Util.PluralSuffix( nFittingsFailed ) ) );

      Util.InfoMsg( string.Format(
        "Set max flow parameter on {0} duct{1}.",
        nDucts, Util.PluralSuffix( nDucts ) ) );
        
      return Result.Succeeded;
    }
  }
}
