﻿using System;
using System.Collections.Generic;
using System.Drawing;

namespace GetLoops
{
  /// <summary>
  /// A closed polygon boundary loop.
  /// </summary>
  class JtLoop : List<Point2dInt>
  {
    public JtLoop( int capacity )
      : base( capacity )
    {
    }

    /// <summary>
    /// Display as a string.
    /// </summary>
    public override string ToString()
    {
      return string.Join( ", ", this );
    }

    /// <summary>
    /// Add another point to the collection.
    /// If the new point is identical to the last,
    /// ignore it. This will automatically suppress
    /// really small boundary segment fragments.
    /// </summary>
    public new void Add( Point2dInt p )
    {
      if( 0 == Count 
        || 0 != p.CompareTo( this[Count - 1] ) )
      {
        base.Add( p );
      }
    }
  }

  /// <summary>
  /// A list of boundary loops.
  /// </summary>
  class JtLoops : List<JtLoop>
  {
    public JtLoops( int capacity )
      : base( capacity )
    {
    }

    /// <summary>
    /// Unite two collections of boundary 
    /// loops into one single one.
    /// </summary>
    public static JtLoops operator+( JtLoops a, JtLoops b )
    {
      int na = a.Count;
      int nb = b.Count;
      JtLoops sum = new JtLoops( na + nb );
      sum.AddRange( a );
      sum.AddRange( b );
      return sum;
    }

    /// <summary>
    /// Return suitable input for the .NET 
    /// GraphicsPath.AddLines method to display the 
    /// loops in a form. Note that a closing segment 
    /// to connect the last point back to the first
    /// is added.
    /// </summary>
    public List<Point[]> GetGraphicsPathLines()
    {
      int i, n;

      List<Point[]> loops 
        = new List<Point[]>( Count );
      
      foreach( JtLoop jloop in this )
      {
        n = jloop.Count;
        Point[] loop = new Point[n + 1];
        i = 0;
        foreach( Point2dInt p in jloop )
        {
          loop[i++] = new Point( p.X, p.Y );
        }
        loop[i] = loop[0];
        loops.Add( loop );
      }
      return loops;
    }
  }
}
