﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Architecture;
using DreamSeat;

namespace GetLoops
{
  class DbUpload
  {
    const string _web_url = "jt.iriscouch.com";
    const string _local_url = "localhost";
    const string _database_name = "rooms";

    /// <summary>
    /// Upload model, level, room and furniture data 
    /// to an IrisCouch hosted CouchDB data repository.
    /// </summary>
    static public void DbUploadRoom(
      Room room,
      List<Element> furniture,
      JtLoops roomLoops,
      Dictionary<string, JtLoop> furnitureLoops,
      List<JtPlacement2dInt> furnitureInstances )
    {
      CouchClient client = new CouchClient(
        _web_url,
        //"localhost", // does not work for me
        //"127.0.0.1", // does not work for me
        //_local_url,
        5984 );

      CouchDatabase db = client.GetDatabase(
        _database_name, true );

      Document doc = room.Document;

      Element projectInfo
        = new FilteredElementCollector( doc )
          .OfClass( typeof( ProjectInfo ) )
          .FirstElement();

      string uid = projectInfo.UniqueId;

      DbModel dbModel;

      if( db.DocumentExists( uid ) )
      {
        dbModel = db.GetDocument<DbModel>( uid );

        Debug.Assert(
          dbModel.Id.Equals( projectInfo.UniqueId ),
          "expected equal ids" );

        dbModel.Description = Util.ElementDescription(
          projectInfo );

        dbModel.Name = projectInfo.Document.Title;

        dbModel = db.UpdateDocument<DbModel>(
          dbModel );
      }
      else
      {
        dbModel = new DbModel();

        dbModel.Id = uid;
        dbModel.Description = Util.ElementDescription(
          projectInfo );

        dbModel.Name = projectInfo.Name;
        dbModel = db.CreateDocument<DbModel>( dbModel );
      }

      Level level = room.Level;

      uid = level.UniqueId;

      DbLevel dbLevel;

      if( db.DocumentExists( uid ) )
      {
        dbLevel = db.GetDocument<DbLevel>( uid );

        Debug.Assert(
          dbLevel.Id.Equals( level.UniqueId ),
          "expected equal ids" );

        dbLevel.Description = Util.ElementDescription(
          level );

        dbLevel.Name = level.Name;
        dbLevel.ModelId = projectInfo.UniqueId;

        dbLevel = db.UpdateDocument<DbLevel>(
          dbLevel );
      }
      else
      {
        dbLevel = new DbLevel();

        dbLevel.Id = uid;
        dbLevel.Description = Util.ElementDescription(
          level );

        dbLevel.Name = level.Name;
        dbLevel.ModelId = projectInfo.UniqueId;

        dbLevel = db.CreateDocument<DbLevel>(
          dbLevel );
      }

      uid = room.UniqueId;

      DbRoom dbRoom;

      if( db.DocumentExists( uid ) )
      {
        dbRoom = db.GetDocument<DbRoom>( uid );

        Debug.Assert(
          dbRoom.Id.Equals( room.UniqueId ),
          "expected equal ids" );

        dbRoom.Description = Util.ElementDescription(
          room );

        dbRoom.Name = room.Name;
        dbRoom.LevelId = level.UniqueId;
        dbRoom.Loops = roomLoops.SvgPath;
        dbRoom.ViewBox = roomLoops.BoundingBox.SvgViewBox;

        dbRoom = db.UpdateDocument<DbRoom>( dbRoom );
      }
      else
      {
        dbRoom = new DbRoom();

        dbRoom.Id = uid;
        dbRoom.Description = Util.ElementDescription(
          room );

        dbRoom.Name = room.Name;
        dbRoom.LevelId = level.UniqueId;
        dbRoom.Loops = roomLoops.SvgPath;
        dbRoom.ViewBox = roomLoops.BoundingBox.SvgViewBox;

        dbRoom = db.CreateDocument<DbRoom>( dbRoom );
      }

      foreach( KeyValuePair<string, JtLoop> p in furnitureLoops )
      {
        uid = p.Key;
        Element e = doc.GetElement( uid );
        if( db.DocumentExists( uid ) )
        {
          DbSymbol symbol = db.GetDocument<DbSymbol>(
            uid );

          symbol.Description = Util.ElementDescription( e );
          symbol.Name = e.Name;
          symbol.Loop = p.Value.SvgPath;

          symbol = db.UpdateDocument<DbSymbol>( symbol );
        }
        else
        {
          DbSymbol symbol = new DbSymbol();
          symbol.Id = uid;
          symbol.Description = Util.ElementDescription( e );
          symbol.Name = e.Name;
          symbol.Loop = p.Value.SvgPath;

          symbol = db.CreateDocument<DbSymbol>( symbol );
        }
      }

      foreach( FamilyInstance f in furniture )
      {
        uid = f.UniqueId;
        if( db.DocumentExists( uid ) )
        {
          DbFurniture dbf = db.GetDocument<DbFurniture>(
            uid );

          dbf.Description = Util.ElementDescription( f );
          dbf.Name = f.Name;
          dbf.RoomId = room.UniqueId;
          dbf.SymbolId = f.Symbol.UniqueId;
          dbf.Transform = new JtPlacement2dInt( f )
            .SvgTransform;

          dbf = db.UpdateDocument<DbFurniture>( dbf );
        }
        else
        {
          DbFurniture dbf = new DbFurniture();
          dbf.Id = uid;
          dbf.Description = Util.ElementDescription( f );
          dbf.Name = f.Name;
          dbf.RoomId = room.UniqueId;
          dbf.SymbolId = f.Symbol.UniqueId;
          dbf.Transform = new JtPlacement2dInt( f )
            .SvgTransform;

          dbf = db.CreateDocument<DbFurniture>( dbf );
        }
      }
    }
  }
}
