//
// (C) Copyright 2003-2007 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE. AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Analysis;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;

namespace Revit.Test.CSTemplate
{
  [Regeneration( RegenerationOption.Manual )]
  [Transaction( TransactionMode.Manual )]
  public class RestoreViewCommand : IExternalCommand
  {
    #region IExternalCommand Members

    public Result Execute( ExternalCommandData commandData, ref string message, ElementSet elements )
    {
      Document doc = commandData.Application.ActiveUIDocument.Document;
      Transaction t = new Transaction( doc, "Restore View" );
      t.Start();
      View view = doc.ActiveView;

      SpatialFieldManager sfm = SpatialFieldManager.GetSpatialFieldManager( view );
      if( sfm != null ) sfm.Clear();

      Categories categories = doc.Settings.Categories;

      HighlightIntersectionsCommand.SetCategoryVisible( categories, BuiltInCategory.OST_Walls, view );
      HighlightIntersectionsCommand.SetCategoryVisible( categories, BuiltInCategory.OST_Columns, view );
      HighlightIntersectionsCommand.SetCategoryVisible( categories, BuiltInCategory.OST_StructuralColumns, view );

      t.Commit();

      return Result.Succeeded;
    }

    #endregion
  }

  /// <summary>
  /// get element's id and type information and its supported information.
  /// </summary>
  [Regeneration( RegenerationOption.Manual )]
  [Transaction( TransactionMode.Manual )]
  public class HighlightIntersectionsCommand : IExternalCommand
  {
    public Result Execute(
      ExternalCommandData revit,
      ref string message,
      ElementSet elements )
    {
      m_uiApp = revit.Application;
      m_app = m_uiApp.Application;
      m_uiDoc = m_uiApp.ActiveUIDocument;
      m_doc = m_uiDoc.Document;

      //TaskDialog.Show("Test application", "Revit version: " + m_app.VersionBuild);

      //m_writer = new StreamWriter( @"C:\CSTemplate.txt" );

      HighlightIntersections( revit.Application.ActiveUIDocument.Document );

      //m_writer.Close();

      return Result.Succeeded;
    }

    /// <summary>
    /// A data structure containing the details of each intersecting wall/column pair.
    /// </summary>
    struct Intersection
    {
      public Element Wall;
      public Element Column;
      public Solid Solid;
    }

    /// <summary>
    /// A collection of all intersections.
    /// </summary>
    private List<Intersection> m_allIntersections;

    /// <summary>
    /// Finds and posts information on wall/column intersections.
    /// </summary>
    /// <param name="doc">The document.</param>
    public void FindIntersectionVolumes( Document doc )
    {
      // Find all Wall elements.
      FilteredElementCollector collector = new FilteredElementCollector( doc );
      collector.OfClass( typeof( FamilyInstance ) );
      m_allIntersections = new List<Intersection>();

      List<BuiltInCategory> categories = new List<BuiltInCategory>();
      categories.Add( BuiltInCategory.OST_Columns );
      categories.Add( BuiltInCategory.OST_StructuralColumns );
      ElementMulticategoryFilter categoryFilter = new ElementMulticategoryFilter( categories );
      collector.WherePasses( categoryFilter );

      foreach( FamilyInstance famInst in collector.OfType<FamilyInstance>() )
      {
        FilteredElementCollector walllIntersectionCollector = new FilteredElementCollector( doc );
        walllIntersectionCollector.OfClass( typeof( Wall ) );

        // Columns may be one of two different categories

        GeometryElement geomElem = famInst.GetOriginalGeometry( new Options() ).GetTransformed( famInst.GetTransform() );

        Solid columnSolid = GetGeometry( geomElem );

        // Apply intersection filter to find matches
        ElementIntersectsSolidFilter intersectsFilter = new ElementIntersectsSolidFilter( columnSolid );
        walllIntersectionCollector.WherePasses( intersectsFilter );

        foreach( Element element in walllIntersectionCollector )
        {
          // Store information on intersection
          Intersection intersection;
          intersection.Wall = element as Wall;
          intersection.Column = famInst;

          GeometryElement wallGeometry = element.get_Geometry( new Options() );
          Solid wallSolid = GetGeometry( wallGeometry );

          // Intersect the solid geometry of the two elements
          intersection.Solid = BooleanOperationsUtils.ExecuteBooleanOperation( wallSolid, columnSolid, BooleanOperationsType.Intersect );

          m_allIntersections.Add( intersection );
        }
      }

      TaskDialog td = new TaskDialog( "Intersection info" );
      td.MainInstruction = "Intersections found: " + m_allIntersections.Count;
      StringBuilder builder = new StringBuilder();
      foreach( Intersection intersection in m_allIntersections )
      {
        builder.AppendLine( String.Format( "{0} x {1}: volume {2} faces {3}", intersection.Wall.Name, intersection.Column.Name, intersection.Solid.Volume, intersection.Solid.Faces.Size ) );
      }
      td.MainContent = builder.ToString();

      td.Show();
    }

    /// <summary>
    /// Return the solid geometry of an element.  
    /// </summary>
    /// <remarks>Makes an assumption that each element consists of only one 
    /// positive-volume solid, and returns the first one it finds.</remarks>
    public Solid GetGeometry( GeometryElement geomElem )
    {
      foreach( GeometryObject geomObj in geomElem.Objects )
      {
        // Walls and some columns will have a solid directly in its geometry
        if( geomObj is Solid )
        {
          Solid solid = ( Solid ) geomObj;
          if( solid.Volume > 0 )
            return solid;
        }

        // Some columns will have a instance pointing to symbol geometry
        if( geomObj is GeometryInstance )
        {
          GeometryInstance geomInst = ( GeometryInstance ) geomObj;
          // Instance geometry is obtained so that the intersection works as
          // expected without requiring transformation
          GeometryElement instElem = geomInst.GetInstanceGeometry();

          foreach( GeometryObject instObj in instElem.Objects )
          {
            if( instObj is Solid )
            {
              Solid solid = ( Solid ) instObj;
              if( solid.Volume > 0 )
                return solid;
            }
          }
        }
      }
      return null;
    }

    public void HighlightIntersections( Document doc )
    {
      FindIntersectionVolumes( doc );

      Transaction t = new Transaction( doc, "Hide categories" );
      t.Start();
      View view = doc.ActiveView;

      Categories categories = doc.Settings.Categories;

      SetCategoryInvisible( categories, BuiltInCategory.OST_Walls, view );
      SetCategoryInvisible( categories, BuiltInCategory.OST_Columns, view );
      SetCategoryInvisible( categories, BuiltInCategory.OST_StructuralColumns, view );

      t.Commit();

      double value = 1.0;
      foreach( Intersection intersection in m_allIntersections )
      {
        PaintSolid( doc, intersection.Solid, value );
        value++;
      }
    }

    public void RestoreViewAfterHighlight( Document doc )
    {
      Transaction t = new Transaction( doc, "Show categories" );
      t.Start();
      View view = doc.ActiveView;

      Categories categories = doc.Settings.Categories;

      SetCategoryVisible( categories, BuiltInCategory.OST_Walls, view );
      SetCategoryVisible( categories, BuiltInCategory.OST_Columns, view );
      SetCategoryVisible( categories, BuiltInCategory.OST_StructuralColumns, view );

      t.Commit();

      SpatialFieldManager sfm = SpatialFieldManager.GetSpatialFieldManager( doc.ActiveView );
      if( sfm != null )
      {
        sfm.Clear();
      }
    }

    private int schemaId = -1;

    private void PaintSolid( Document doc, Solid s, double value )
    {
      Application app = doc.Application;

      View view = doc.ActiveView;

      if( view.AnalysisDisplayStyleId == ElementId.InvalidElementId )
      {
        CreateAVFDisplayStyle( doc, view );
      }

      SpatialFieldManager sfm = SpatialFieldManager.GetSpatialFieldManager( view );
      if( sfm == null ) sfm = SpatialFieldManager.CreateSpatialFieldManager( view, 1 );

      if( schemaId != -1 )
      {
        IList<int> results = sfm.GetRegisteredResults();

        if( !results.Contains( schemaId ) )
        {
          schemaId = -1;
        }
      }

      if( schemaId == -1 )
      {
        AnalysisResultSchema resultSchema1 = new AnalysisResultSchema( "PaintedSolid", "Description" );
        schemaId = sfm.RegisterResult( resultSchema1 );
      }

      FaceArray faces = s.Faces;
      Transform trf = Transform.Identity;

      foreach( Face face in faces )
      {
        int idx = sfm.AddSpatialFieldPrimitive( face, trf );

        IList<UV> uvPts = new List<UV>();
        List<double> doubleList = new List<double>();
        IList<ValueAtPoint> valList = new List<ValueAtPoint>();
        BoundingBoxUV bb = face.GetBoundingBox();
        uvPts.Add( bb.Min );
        doubleList.Add( value );
        valList.Add( new ValueAtPoint( doubleList ) );
        FieldDomainPointsByUV pnts = new FieldDomainPointsByUV( uvPts );
        FieldValues vals = new FieldValues( valList );

        sfm.UpdateSpatialFieldPrimitive( idx, pnts, vals, schemaId );
      }
    }

    private void CreateAVFDisplayStyle( Document doc, View view )
    {
      Transaction t = new Transaction( doc, "Create AVF style" );
      t.Start();
      AnalysisDisplayColoredSurfaceSettings coloredSurfaceSettings = new AnalysisDisplayColoredSurfaceSettings();
      coloredSurfaceSettings.ShowGridLines = true;
      AnalysisDisplayColorSettings colorSettings = new AnalysisDisplayColorSettings();
      AnalysisDisplayLegendSettings legendSettings = new AnalysisDisplayLegendSettings();
      legendSettings.ShowLegend = false;
      AnalysisDisplayStyle analysisDisplayStyle =
          AnalysisDisplayStyle.CreateAnalysisDisplayStyle( doc, "Paint Solid", coloredSurfaceSettings, colorSettings, legendSettings );

      view.AnalysisDisplayStyleId = analysisDisplayStyle.Id;
      t.Commit();
    }

    public static void SetCategoryInvisible( Categories categories, BuiltInCategory bic, View view )
    {
      SetCategoryVisibility( categories, bic, view, false );
    }

    public static void SetCategoryVisible( Categories categories, BuiltInCategory bic, View view )
    {
      SetCategoryVisibility( categories, bic, view, true );
    }

    private static void SetCategoryVisibility( Categories categories, BuiltInCategory bic, View view, bool visible )
    {
      Category category = categories.get_Item( bic );
      category.set_Visible( view, visible );
    }

    private Autodesk.Revit.UI.UIApplication m_uiApp;
    private Autodesk.Revit.UI.UIDocument m_uiDoc;
    private Application m_app;
    private Autodesk.Revit.DB.Document m_doc;

    //private TextWriter m_writer;
  }
}
