#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Structure;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using RvtOperationCanceledException = Autodesk.Revit.Exceptions.OperationCanceledException;
#endregion

namespace GetBeamColumns
{
  [Transaction( TransactionMode.ReadOnly )]
  public class Command : IExternalCommand
  {
    const double _eps = 0.1e-9;

    double SignedDistanceTo( Plane plane, XYZ p )
    {
      XYZ v = plane.Normal;
      return v.DotProduct( p )
        - v.DotProduct( plane.Origin );
    }

    class BeamPickFilter : ISelectionFilter
    {
      public bool AllowElement( Element e )
      {
        return null != e.Category
          && e.Category.Id.IntegerValue.Equals(
            (int) BuiltInCategory.OST_StructuralFraming );
      }

      public bool AllowReference( Reference r, XYZ p )
      {
        return true;
      }
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;
      Element beam = null;

      try
      {
        Selection sel = uidoc.Selection;

        Reference r = sel.PickObject(
          ObjectType.Element,
          new BeamPickFilter(),
          "Please select a beam" );

        beam = doc.GetElement( r );
      }
      catch( RvtOperationCanceledException )
      {
        return Result.Cancelled;
      }

      List<ElementId> columnIds = null;

      // Optionally switch between different tests
      // by modifying these values in the debugger

      bool useBoundingBox = false;
      bool useSolid = false;
      bool useElement = false;
      bool useGeneratingIds = false;
      bool useLocation = true;

      #region Obsolete previous attempts
      if( useBoundingBox )
      {
        BoundingBoxXYZ box = beam.get_BoundingBox( null );

        Outline outline = new Outline( box.Min, box.Max );

        ElementFilter bbfilter = new BoundingBoxIntersectsFilter(
          outline, 0.1 );

        FilteredElementCollector columns
          = new FilteredElementCollector( doc )
            .WhereElementIsNotElementType()
            .OfCategory( BuiltInCategory.OST_StructuralColumns )
            .WherePasses( bbfilter );
      }

      if( useSolid )
      {
        Options opt = app.Create.NewGeometryOptions();
        GeometryElement geo = beam.get_Geometry( opt );
        Solid solid = null;

        //geo = geo.GetTransformed(
        //  Transform.Identity );

        //GeometryInstance inst = null;

        //foreach( GeometryObject obj in geo )
        //{
        //  inst = obj as GeometryInstance;

        //  if( null != inst ) { break; }
        //}

        //geo = inst.GetSymbolGeometry();

        foreach( GeometryObject obj in geo )
        {
          solid = obj as Solid;

          if( null != solid
            && 0 < solid.Faces.Size )
          {
            break;
          }
        }

#if NEED_TO_RECREATE_SOLID
      // Determine beam location curve for 
      // extrusion direction and length

      LocationCurve lc = beam.Location as LocationCurve;

      Curve curve = lc.Curve;
      XYZ p = curve.get_EndPoint( 0 );
      XYZ q = curve.get_EndPoint( 1 );
      XYZ v = q - p;

      // Determine beam start face for extrusion profile

      Face startFace = null;

      double area = double.MaxValue;
      double dist = double.MaxValue;

      foreach( Face f in solid.Faces )
      {
        PlanarFace pf = f as PlanarFace;

        if( null != pf )
        {
          Plane plane
            = new Plane( pf.Normal, pf.Origin );

          double d = Math.Abs( SignedDistanceTo( plane, p ) );

          if( d - _eps < dist )
          {
            if( f.Area - _eps < area )
            {
              startFace = f;
              area = f.Area;
              dist = d;
            }
          }
        }
      }

      // Determine profile loops representing 
      // beam cross section.

      n = startFace.EdgeLoops.Size;

      Debug.Assert( 1 == n, "expected just one single beam cross section profile loop" );

      IList<CurveLoop> profileLoops = new List<CurveLoop>( n );
      IList<Curve> loop = new List<Curve>();

      foreach( EdgeArray ea in startFace.EdgeLoops )
      {
        loop.Clear();

        foreach( Edge edge in ea )
        {
          loop.Add( edge.AsCurve() );
        }
        profileLoops.Add( CurveLoop.Create( loop ) );
      }

      Debug.Assert( 1 == profileLoops.Count, "expected just one profile loop" );
      Debug.Assert( profileLoops[0].HasPlane(), "expected planar profile loop" );
      Debug.Assert( _eps > Math.Abs( SignedDistanceTo( profileLoops[0].GetPlane(), p ) ), "expected planar profile to contain beam location curve start point" );

      // Extrude beam cross section 
      // along beam location curve.

      solid = GeometryCreationUtilities
        .CreateExtrusionGeometry(
          profileLoops, v, v.GetLength() );
#endif // NEED_TO_RECREATE_SOLID

        ElementFilter beamIntersectFilter
          = new ElementIntersectsSolidFilter( solid );

        FilteredElementCollector columns
          = new FilteredElementCollector( doc )
            .WhereElementIsNotElementType()
            .OfCategory( BuiltInCategory.OST_StructuralColumns )
            .WherePasses( beamIntersectFilter );
      }

      if( useElement )
      {
        // Initially, no columns are found to 
        // intersect the beam. Maybe it will help to
        // move the beam down a bit?

        using( TransactionGroup txg = new TransactionGroup( doc ) )
        {
          txg.Start( "Find Columns Intersecting Beam" );

          using( Transaction tx = new Transaction( doc ) )
          {
            tx.Start( "Temporarily Move Beam Down a Little" );

            ElementTransformUtils.MoveElement(
              doc, beam.Id, -0.1 * XYZ.BasisZ );

            tx.Commit();
          }

          ElementFilter beamIntersectFilter
            = new ElementIntersectsElementFilter( beam );

          FilteredElementCollector columns
            = new FilteredElementCollector( doc )
              .WhereElementIsNotElementType()
              .OfCategory( BuiltInCategory.OST_StructuralColumns )
              .WherePasses( beamIntersectFilter );

          columnIds = new List<ElementId>(
            columns.ToElementIds() );

          // We do not commit the transaction group, 
          // because no modifications should be saved.
          // The transaction group is only created and 
          // started to encapsulate the transactions 
          // required by the IsolateElementTemporary 
          // method. Since the transaction group is not 
          // committed, the changes are automatically 
          // discarded.

          //txg.Commit();
        }
      }

      if( useGeneratingIds )
      {
        Options opt = app.Create.NewGeometryOptions();
        GeometryElement geo = beam.get_Geometry( opt );

        foreach( GeometryObject obj in geo )
        {
          Solid solid = obj as Solid;

          if( null != solid )
          {
            foreach( Face f in solid.Faces )
            {
              ICollection<ElementId> ids
                = beam.GetGeneratingElementIds( f );

              foreach( ElementId id in ids )
              {
                Element e = doc.GetElement( id );
                if( null != e.Category
                  && e.Category.Id.IntegerValue.Equals(
                    (int) BuiltInCategory.OST_StructuralColumns ) )
                {
                  columnIds.Add( id );
                }
              }
            }
          }
        }
      }
      #endregion // Obsolete previous attempts

      if( useLocation )
      {
        // Determine beam location curve for 
        // extrusion direction and length

        LocationCurve lc = beam.Location as LocationCurve;

        Curve curve = lc.Curve;

        Solid solid = null;

        // Handle generic curve parameters.
        // See below for simplified linear case.

        XYZ p = curve.get_EndPoint( 0 );
        double param = curve.get_EndParameter( 0 );
        Transform transform = curve.ComputeDerivatives( param, false );
        Debug.Assert( p.IsAlmostEqualTo( transform.Origin ),
          "expected derivative origin to equal evaluation curve point" );
        XYZ tangent = transform.BasisX;

        // Use bounding box to determine elevation of
        // bottom of beam and how far downwards to 
        // offset location line -- one inch below 
        // beam bottom.

        BoundingBoxXYZ bb = beam.get_BoundingBox( null );

        Debug.Assert( .001 > bb.Min.Z - bb.Max.Z,
          "expected horizontal beam" );

        double inch = 1.0 / 12.0;
        double beamBottom = bb.Min.Z;

        XYZ arcCenter = new XYZ( p.X, p.Y,
          beamBottom - inch );

        Plane plane = new Plane( tangent, arcCenter );

        CurveLoop profileLoop = new CurveLoop();

        Autodesk.Revit.Creation.Application creapp
          = app.Create;

        Arc arc1 = creapp.NewArc(
          plane, inch, 0, Math.PI );

        Arc arc2 = creapp.NewArc(
          plane, inch, Math.PI, 2 * Math.PI );

        profileLoop.Append( arc1 );
        profileLoop.Append( arc2 );

        List<CurveLoop> loops = new List<CurveLoop>( 1 );
        loops.Add( profileLoop );

        // Switch this on to handle a straight beam as 
        // a separate simplified case using 
        // CreateExtrusionGeometry instead of the 
        // generic CreateSweptGeometry solution.

        bool checkForLine = false;

        if( checkForLine && curve is Line )
        {
          XYZ q = curve.get_EndPoint( 1 );
          XYZ v = q - p;

          Debug.Assert( 0.01 > v.Z,
            "expected horizontal beam" );

          Debug.Assert( v.IsAlmostEqualTo( tangent ),
            "expected straight beam vector to equal start tangent" );

          solid = GeometryCreationUtilities
            .CreateExtrusionGeometry( loops, v, v.GetLength() );
        }
        else
        {
          // Offset location curve downward 
          // one inch  below beam bottom face

          XYZ offset = arcCenter - p;

          transform = Transform.get_Translation(
            offset );

          CurveLoop sweepPath = new CurveLoop();

          sweepPath.Append( curve.get_Transformed(
            transform ) );

          solid = GeometryCreationUtilities
            .CreateSweptGeometry(
              sweepPath, 0, param, loops );
        }

        ElementFilter beamIntersectFilter
          = new ElementIntersectsSolidFilter( solid );

        columnIds = new List<ElementId>(
          new FilteredElementCollector( doc )
            .WhereElementIsNotElementType()
            .OfCategory( BuiltInCategory.OST_StructuralColumns )
            .WherePasses( beamIntersectFilter )
            .ToElementIds() );
      }

      int n = ( null == columnIds )
        ? 0
        : columnIds.Count<ElementId>();

      string s1 = string.Format(
        "Selected beam is supported by {0} column{1}{2}",
        n,
        ( 1 == n ? "" : "s" ),
        ( 0 == n ? "." : ":" ) );

      string s2 = "<None>";

      if( 0 < n )
      {
        uidoc.Selection.Elements.Clear();

        foreach( ElementId id in columnIds )
        {
          Element e = doc.GetElement( id );
          uidoc.Selection.Elements.Add( e );
        }

        s2 = string.Join( ", ",
          columnIds.ConvertAll<string>(
            id => id.IntegerValue.ToString() ) );
      }

      TaskDialog.Show( s1, s2 );

      return Result.Succeeded;
    }
  }

  #region AnisCommand
  [Transaction( TransactionMode.Manual )]
  public class AnisCommand : IExternalCommand
  {
    class BeamPickFilter : ISelectionFilter
    {
      public bool AllowElement( Element e )
      {
        return e.Category.Id.IntegerValue.Equals( -2001320 );
      }

      public bool AllowReference( Reference r, XYZ p )
      {
        return false;
      }
    }

    IList<Element> GetListOfCoulumnsTouchingBeam(
      Element beam,
      Document doc )
    {
      ElementStructuralTypeFilter filter = new ElementStructuralTypeFilter( StructuralType.Column );
      FilteredElementCollector collector = new FilteredElementCollector( doc );
      FilteredElementCollector collector2 = collector.WherePasses( filter ).WhereElementIsNotElementType();
      BoundingBoxXYZ xxyz = beam.get_BoundingBox( doc.ActiveView );
      Outline outline = new Outline( xxyz.Min, xxyz.Max );
      BoundingBoxIntersectsFilter filter2 = new BoundingBoxIntersectsFilter( outline );
      return collector2.WherePasses( filter2 ).ToElements();
    }

    public Result Execute( ExternalCommandData commandData, ref string message, ElementSet elements )
    {
      UIDocument uidoc = commandData.Application.ActiveUIDocument;
      Document document = uidoc.Document;

      Reference r = uidoc.Selection.PickObject(
        ObjectType.Element, new BeamPickFilter(),
        "Select a beam" );

      Element beam = document.GetElement( r );

      IList<Element> listOfCoulumnsTouchingBeam
        = GetListOfCoulumnsTouchingBeam(
          beam, document );

      StringBuilder builder = new StringBuilder();

      foreach( Element e in listOfCoulumnsTouchingBeam )
      {
        uidoc.Selection.Elements.Add( e );
        builder.AppendLine( e.Id.ToString() );
      }
      TaskDialog.Show( "BB", builder.ToString() );

      return Result.Succeeded;
    }
  }
  #endregion // AnisCommand
}
