#region Namespaces
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Architecture;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using BoundarySegment = Autodesk.Revit.DB.BoundarySegment;
#endregion

namespace GetLoops
{
  [Transaction( TransactionMode.ReadOnly )]
  public class Command : IExternalCommand
  {
    #region RoomSelectionFilter
    class RoomSelectionFilter : ISelectionFilter
    {
      public bool AllowElement( Element e )
      {
        return e is Room;
      }

      public bool AllowReference( Reference r, XYZ p )
      {
        return true;
      }
    }
    #endregion // RoomSelectionFilter

    const string _caption = "Export Room and Equipment Boundary Loops";

    /// <summary>
    /// Conversion factor from foot to quarter inch.
    /// </summary>
    const double _quarter_inch = 1.0 / (12 * 4);

    static void InfoMsg( string msg )
    {
      Debug.Print( msg );
      TaskDialog.Show( _caption, msg );
    }

    static void ErrorMsg( string msg )
    {
      Debug.Print( msg );
      TaskDialog dlg = new TaskDialog( _caption );
      dlg.MainIcon = TaskDialogIcon.TaskDialogIconWarning;
      dlg.MainInstruction = msg;
      dlg.Show();
    }

    /// <summary>
    /// Retrieve the room plan view boundary 
    /// polygon loops and convert to 2D integer-based.
    /// For optimisation and consistency reasons, 
    /// convert all coordinates to integer values in
    /// millimetres. Revit precision is limited to 
    /// 1/16 of an inch, which is abaut 1.2 mm, anyway.
    /// </summary>
    JtLoops GetRoomLoops( Room room )
    {
      SpatialElementBoundaryOptions opt 
        = new SpatialElementBoundaryOptions();

      opt.SpatialElementBoundaryLocation =
        SpatialElementBoundaryLocation.Center; // loops closed
        //SpatialElementBoundaryLocation.Finish; // loops not closed

      IList<IList<BoundarySegment>> loops = room.
        GetBoundarySegments( opt );

      int nLoops = loops.Count;

      JtLoops jtloops = new JtLoops( nLoops );

      foreach( IList<BoundarySegment> loop in loops )
      {
        int nSegments = loop.Count;

        JtLoop jtloop = new JtLoop( nSegments );

        XYZ p0 = null; // loop start point
        XYZ p; // segment start point
        XYZ q = null; // segment end point

        foreach( BoundarySegment seg in loop )
        {
          p = seg.Curve.get_EndPoint( 0 );

          jtloop.Add( new Point2dInt( p ) );

          Debug.Assert( null == q || q.IsAlmostEqualTo( p ),
            "expected last endpoint to equal current start point" );

          q = seg.Curve.get_EndPoint( 1 );
          
          Debug.Print( "{0} --> {1}",
            Util.PointString( p ),
            Util.PointString( q ) );

          if( null == p0 )
          {
            p0 = p; // save loop start point
          }
        }
        Debug.Assert( q.IsAlmostEqualTo( p0 ),
          "expected last endpoint to equal loop start point" );

        jtloops.Add( jtloop );
      }
      return jtloops;
    }

    //(9.03,10.13,0) --> (-14.59,10.13,0)
    //(-14.59,10.13,0) --> (-14.59,1.93,0)
    //(-14.59,1.93,0) --> (-2.45,1.93,0)
    //(-2.45,1.93,0) --> (-2.45,-3.98,0)
    //(-2.45,-3.98,0) --> (9.03,-3.98,0)
    //(9.03,-3.98,0) --> (9.03,10.13,0)
    //(0.98,-0.37,0) --> (0.98,1.93,0)
    //(0.98,1.93,0) --> (5.57,1.93,0)
    //(5.57,1.93,0) --> (5.57,-0.37,0)
    //(5.57,-0.37,0) --> (0.98,-0.37,0)

    //(9.03,10.13) --> (-14.59,10.13)
    //(-14.59,10.13) --> (-14.59,1.93)
    //(-14.59,1.93) --> (-2.45,1.93)
    //(-2.45,1.93) --> (-2.45,-3.98)
    //(-2.45,-3.98) --> (9.03,-3.98)
    //(9.03,-3.98) --> (9.03,10.13)
    //(0.98,-0.37) --> (0.98,1.93)
    //(0.98,1.93) --> (5.57,1.93)
    //(5.57,1.93) --> (5.57,-0.37)
    //(5.57,-0.37) --> (0.98,-0.37)

    //Room Rooms <212639 Room 1> has 2 loops:
    //  0: (2753,3087), (-4446,3087), (-4446,587), (-746,587), (-746,-1212), (2753,-1212)
    //  1: (298,-112), (298,587), (1698,587), (1698,-112)

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Application app = uiapp.Application;
      Document doc = uidoc.Document;

      if( null == doc )
      {
        ErrorMsg( "Please run this command in a valid"
          + " Revit project document." );
        return Result.Failed;
      }

      // Iterate over all pre-selected rooms

      List<ElementId> ids = null;

      Selection sel = uidoc.Selection;

      if( 0 < sel.Elements.Size )
      {
        foreach( Element e in sel.Elements )
        {
          if( !( e is Room ) )
          {
            ErrorMsg( "Please pre-select only room"
              + " elements before running this command." );
            return Result.Failed;
          }

          if( null == ids )
          {
            ids = new List<ElementId>( 1 );
          }

          ids.Add( e.Id );
        }
      }

      // If no rooms were pre-selected, 
      // prompt for post-selection

      if( null == ids )
      {
        IList<Reference> refs = null;

        try
        {
          refs = sel.PickObjects( ObjectType.Element,
            new RoomSelectionFilter(),
            "Please select rooms." );
        }
        catch( Autodesk.Revit.Exceptions
          .OperationCanceledException )
        {
          return Result.Cancelled;
        }
        ids = new List<ElementId>(
          refs.Select<Reference, ElementId>(
            r => r.ElementId ) );
      }

      foreach( ElementId id in ids )
      {
        Element e = doc.GetElement( id );

        Debug.Assert( e is Room,
          "expected parts only" );

        JtLoops roomLoops = GetRoomLoops( e as Room );

        int nLoops = roomLoops.Count;

        Debug.Print( "{0} has {1} loop{2}{3}",
          Util.ElementDescription( e ), nLoops,
          Util.PluralSuffix( nLoops ), 
          Util.DotOrColon( nLoops ) );

        int i = 0;

        foreach( JtLoop loop in roomLoops )
        {
          Debug.Print( "  {0}: {1}", i++, loop.ToString() );
        }
      }
      return Result.Succeeded;
    }
  }
}
