/* LRUG Presentation on 16.03.2011 by Ritchie Jackson
 * "Revit 2011 API - an Introduction"
 *  NOTE:
 *  All code provided on an 'as-is' basis
 *  No error-checking provided in order to 'keep-it-simple'
 *  The macros should be run in a Generic Model Family document
 */

using System;
using System.Collections.Generic;
// Provide access to Revit functions
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;

namespace LRUG_01_Basic
{
    [Autodesk.Revit.Attributes.Transaction(Autodesk.Revit.Attributes.TransactionMode.Manual)]
    [Autodesk.Revit.Attributes.Regeneration(Autodesk.Revit.Attributes.RegenerationOption.Manual)]
    [Autodesk.Revit.VSTA.AddInId("ae2a04d4-9a61-4f25-84d7-e771b1a20e3a")]
    public partial class ThisApplication
    {
        private void Module_Startup(object sender, EventArgs e)
        {
            
        }

        private void Module_Shutdown(object sender, EventArgs e)
        {

        }

        #region VSTA generated code
        private void InternalStartup()
        {
            this.Startup += new System.EventHandler(Module_Startup);
            this.Shutdown += new System.EventHandler(Module_Shutdown);
        }
        #endregion

        
        
        // User-generated code : Draw a single line
        public void Basic_01_Line()
        {
            // Begin a series of instructions to Revit
            Transaction trans = new Transaction(ActiveUIDocument.Document, "Line");
            trans.Start();

            // Define the Document into which the geometry will be placed
            Document doc = ActiveUIDocument.Document;

            // Revit uses Imperial Feet as its internal definition for length -
            // So provide a conversion variable to work in Millimetres:-
            double ftMM = 1 / 304.8;

            // Define two points on the Line ->
            // multiply their coordinates by the conversion variable
            XYZ point01 = new XYZ(); // Defaults to (0.0, 0.0, 0.0) if no values are provided
            XYZ point02 = new XYZ(0.0, 2100.0, 0.0) * ftMM;

            // Create the Line geometry ->
            // 'true' indicates that it terminates at the setout points
            Line line01 = doc.Application.Create.NewLine(point01, point02, true);

            // Define the Plane for the Line placement - X-Axis, Y-Axis, Origin
            Plane plane = doc.Application.Create.NewPlane(XYZ.BasisX, XYZ.BasisY, point01);

            // Create a Sketch Plane from this Plane in order to display the Line
            SketchPlane planeSK = doc.FamilyCreate.NewSketchPlane(plane);

            // Display the Line
            ModelLine line01M = doc.FamilyCreate.NewModelCurve(line01, planeSK) as ModelLine;

            // End the series of instructions
            trans.Commit();
        }

        // User-generated code : Create an Extrusion from one Profile
        public void Basic_02_Extrusion()
        {
            // Begin a series of instructions to Revit
            Transaction trans = new Transaction(ActiveUIDocument.Document, "Extrusion");
            trans.Start();

            // Define the Document into which the geometry will be placed
            Document doc = ActiveUIDocument.Document;

            // Revit uses Imperial Feet as its internal definition for length -
            // So provide a conversion variable to work in Millimetres:-
            double ftMM = 1 / 304.8;

            // Define a Variable to hold the Extrusion Thickness - a single Parameter
            double thickness = 50.0 * ftMM;

            // Define Profile End-Points - multiply their coordinates by the conversion variable
            XYZ point01 = new XYZ(); // Defaults to (0.0, 0.0, 0.0) if no values are provided 
            XYZ point02 = new XYZ(0.0, 2100.0, 0.0) * ftMM;
            XYZ point03 = new XYZ(300.0, 2100.0, 0.0) * ftMM;
            XYZ point04 = new XYZ(300.0, 0.0, 0.0) * ftMM;

            // Create the Profile Edges - they form a single closed-loop
            Line line01 = doc.Application.Create.NewLine(point01, point02, true);
            Line line02 = doc.Application.Create.NewLine(point02, point03, true);
            Line line03 = doc.Application.Create.NewLine(point03, point04, true);
            Line line04 = doc.Application.Create.NewLine(point04, point01, true);

            // Create an Array to hold the closed-loop of Profile Edges
            CurveArray curveAr1 = new CurveArray();

            // Place the single closed-loop of Profile Edges in the Array
            curveAr1.Append(line01);
            curveAr1.Append(line02);
            curveAr1.Append(line03);
            curveAr1.Append(line04);

            // Create an Array of Arrays - allows for more than one closed-loop to be used
            CurveArrArray curveArAr = new CurveArrArray();

            // Place all the closed-loops in the Array of Arrays - only one in this case
            curveArAr.Append(curveAr1);

            // Define the Plane for the Extrusion Profile placement
            Plane plane = doc.Application.Create.NewPlane(curveAr1);

            // Create a Sketch Plane from this Plane
            SketchPlane planeSK = doc.FamilyCreate.NewSketchPlane(plane);

            // Create the Extrusion - 'true' indicates it's Solid
            Extrusion extrude = doc.FamilyCreate.NewExtrusion(true, curveArAr, planeSK, thickness);

            // End the series of instructions
            trans.Commit();
        }

        // User-generated code : Create an Extrusion from two Profiles
        public void Basic_03_Extrusion()
        {
            // Begin a series of instructions to Revit
            Transaction trans = new Transaction(ActiveUIDocument.Document, "Extrusion+Cutout");
            trans.Start();

            // Define the Document into which the geometry will be placed
            Document doc = ActiveUIDocument.Document;

            // Revit uses Imperial Feet as its internal definition for length -
            // So provide a conversion variable to work in Millimetres:-
            double ftMM = 1 / 304.8;

            // Define Variables to hold the Extrusion dimensions - a set of Parameters
            double thickness = 50.0 * ftMM;
            double width = 300.0 * ftMM;
            double length = 2100.0 * ftMM;

            // Define Profile End-Points - unit conversion has already been done in the Parameters
            XYZ point01 = new XYZ(); // Defaults to (0.0, 0.0, 0.0) if no values are provided 
            XYZ point02 = new XYZ(0.0, length, 0.0);
            XYZ point03 = new XYZ(width, length, 0.0);
            XYZ point04 = new XYZ(width, 0.0, 0.0);

            // Define an edge offset for the cutout
            double offSet = 25.0 * ftMM;

            // Create the Profile Edges - they form a single closed-loop
            Line line01 = doc.Application.Create.NewLine(point01, point02, true);
            Line line02 = doc.Application.Create.NewLine(point02, point03, true);
            Line line03 = doc.Application.Create.NewLine(point03, point04, true);
            Line line04 = doc.Application.Create.NewLine(point04, point01, true);

            // Create an Array to hold the closed-loop of Profile Edges
            CurveArray curveAr1 = new CurveArray();

            // Place the single closed-loop of Profile Edges in the Array
            curveAr1.Append(line01);
            curveAr1.Append(line02);
            curveAr1.Append(line03);
            curveAr1.Append(line04);

            // Define 2nd Profile End-Points - unit conversion has already been done in the Parameters
            XYZ point05 = new XYZ(offSet, offSet, 0.0);
            XYZ point06 = new XYZ(offSet, length / 2.0, 0.0);
            XYZ point07 = new XYZ(width - offSet, length / 2.0, 0.0);
            XYZ point08 = new XYZ(width - offSet, offSet, 0.0);
            
            // Create the 2nd Profile Edges - they form a single closed-loop
            Line line05 = doc.Application.Create.NewLine(point05, point06, true);
            Line line06 = doc.Application.Create.NewLine(point06, point07, true);
            Line line07 = doc.Application.Create.NewLine(point07, point08, true);
            Line line08 = doc.Application.Create.NewLine(point08, point05, true);

            // Create an Array to hold the 2nd closed-loop of Profile Edges
            CurveArray curveAr2 = new CurveArray();

            // Place the single closed-loop of Profile Edges in the Array
            curveAr2.Append(line05);
            curveAr2.Append(line06);
            curveAr2.Append(line07);
            curveAr2.Append(line08);
            
            // Create an Array of Arrays - allows for more than one closed-loop to be used
            CurveArrArray curveArAr = new CurveArrArray();

            // Place all the closed-loops in the Array of Arrays - two in this case
            curveArAr.Append(curveAr1);
            curveArAr.Append(curveAr2);

            // Define the Plane for the Extrusion Profile placement - use one of the closed-loop Profiles
            Plane plane = doc.Application.Create.NewPlane(curveAr1);

            // Create a Sketch Plane from this Plane
            SketchPlane planeSK = doc.FamilyCreate.NewSketchPlane(plane);

            // Create the Extrusion - 'true' indicates it's Solid
            Extrusion extrude = doc.FamilyCreate.NewExtrusion(true, curveArAr, planeSK, thickness);

            // End the series of instructions
            trans.Commit();
        }

        // User-generated code : Create a sequence of Extrusions : Requires the 'drawPlank' method below
        public void Basic_04_BoardWalk()
        {
            // Begin a series of instructions to Revit
            Transaction trans = new Transaction(ActiveUIDocument.Document, "BoardWalk");
            trans.Start();

            // Define the Document into which the geometry will be placed
            Document doc = ActiveUIDocument.Document;

            // Revit uses Imperial Feet as its internal definition for length -
            // So provide a conversion variable to work in Millimetres:-
            double ftMM = 1 / 304.8;

            // Define the Interpolation Points for a Spline controlling one edge of the Boardwalk
            XYZ pointB01 = new XYZ(0.0, 3000.0, 0.0) * ftMM;
            XYZ pointB02 = new XYZ(3000.0, 4500.0, 0.0) * ftMM;
            XYZ pointB03 = new XYZ(6000.0, 2100.0, 0.0) * ftMM;
            XYZ pointB04 = new XYZ(9000.0, 2400.0, 0.0) * ftMM;
            XYZ pointB05 = new XYZ(12000.0, 1200.0, 0.0) * ftMM;

            // Define a List to hold the Spline Points
            IList<XYZ> spline1Pnts = new List<XYZ>();
            // Add the Points to this List
            spline1Pnts.Add(pointB01);
            spline1Pnts.Add(pointB02);
            spline1Pnts.Add(pointB03);
            spline1Pnts.Add(pointB04);
            spline1Pnts.Add(pointB05);

            // Create the Spline
            HermiteSpline spline1 = doc.Application.Create.NewHermiteSpline(spline1Pnts, false);

            // Define Variables to hold the Extrusion dimensions - a set of Parameters
            double thickness = 50.0 * ftMM;
            double width = 250.0 * ftMM;
            // 'length' has been omitted as it's now dependent on the Plank's position on the x-Axis

            // Define an edge offset for the cutout
            double edgeOffSet = 50.0 * ftMM;

            // Define the Plane for the Extrusion Profile placements - Origin and z-Axis as the normal
            Plane plane = doc.Application.Create.NewPlane(XYZ.BasisZ, XYZ.Zero);

            // Create a Sketch Plane from this Plane
            SketchPlane planeSK = doc.FamilyCreate.NewSketchPlane(plane);

            // Display the Spline - useful for a visual check during debugging but not required
            ModelCurve spline1M = doc.FamilyCreate.NewModelCurve(spline1, planeSK);

            // Define the Plank spacing
            double plankSpace = (width + 25.0 * ftMM);
            // Determine the number of required Planks ->
            // Overall length is just the x-Value of tbe last point on the Spline
            int numberOfPlanks = (int)(pointB05.X / plankSpace);
            // Create a series of Extrusions by using the 'drawPlank' method
            for (int i = 0; i < numberOfPlanks; i++)
            {
                // x-Offset for the bottom-left corner of the current Plank
                double start = i * plankSpace;
                // Start point of a Line coincident with the Planks's left edge
                XYZ head1 = new XYZ(start, 0.0, 0.0);
                // Start point of a Line coincident with the Plank's right edge
                XYZ head2 = new XYZ(start + width, 0.0, 0.0);
                // Required direction vector of the Line -> vertical
                XYZ direction = new XYZ(0.0, 1.0, 0.0);
                // Create the unbounded Lines
                Line intersectLine1 = doc.Application.Create.NewLine(head1, direction, false);
                Line intersectLine2 = doc.Application.Create.NewLine(head2, direction, false);
                // Find the Line / Spline intersections
                IntersectionResultArray results = new IntersectionResultArray();
                SetComparisonResult result = new SetComparisonResult();
                result = intersectLine1.Intersect(spline1, out results);
                IntersectionResult iResult = new IntersectionResult();
                iResult = results.get_Item(0);
                XYZ plankEndPoint1 = iResult.XYZPoint;
                result = intersectLine2.Intersect(spline1, out results);
                iResult = results.get_Item(0);
                XYZ plankEndPoint2 = iResult.XYZPoint;
                // Send the information for constructing a single Plank to the 'drawPlank' method below
                drawPlank(doc, thickness, width, plankEndPoint1.Y, plankEndPoint2.Y, edgeOffSet, plankEndPoint1.X, planeSK);
            }
            
            // End the series of instructions
            trans.Commit();
        }
        // User-generated code : Create a Plank for the Boardwalk with a given length at a given position on the x-Axis
        // This is not 'stand-alone' code - it is called from the 'Basic_04_BoardWalk' Macro
        private void drawPlank(Document doc, double  height, double width, double length1, double length2,
                                    double offSet, double xPosition, SketchPlane planeSK)
        {
            // Determine vertical offset for each Plank -> proportional to its length
            double vertOffset = length1 / 2.0;

            // Define Profile End-Points - unit conversion has already been done in the Parameters
            XYZ point01 = new XYZ(xPosition, 0.0, vertOffset);
            XYZ point02 = new XYZ(xPosition, length1, vertOffset);
            XYZ point03 = new XYZ(width + xPosition, length2, vertOffset);
            XYZ point04 = new XYZ(width + xPosition, 0.0, vertOffset);

            // Create the Profile Edges - they form a single closed-loop
            Line line01 = doc.Application.Create.NewLine(point01, point02, true);
            Line line02 = doc.Application.Create.NewLine(point02, point03, true);
            Line line03 = doc.Application.Create.NewLine(point03, point04, true);
            Line line04 = doc.Application.Create.NewLine(point04, point01, true);

            // Create an Array to hold the closed-loop of Profile Edges
            CurveArray curveAr1 = new CurveArray();

            // Place the single closed-loop of Profile Edges in the Array
            curveAr1.Append(line01);
            curveAr1.Append(line02);
            curveAr1.Append(line03);
            curveAr1.Append(line04);

            // Define 2nd Profile End-Points - unit conversion has already been done in the Parameters
            XYZ point05 = new XYZ(offSet + xPosition, offSet, vertOffset);
            XYZ point06 = new XYZ(offSet + xPosition, length1 / 3.0, vertOffset);
            XYZ point07 = new XYZ(width - offSet + xPosition, length2 / 3.0, vertOffset);
            XYZ point08 = new XYZ(width - offSet + xPosition, offSet, vertOffset);

            // Create the 2nd Profile Edges - they form a single closed-loop
            Line line05 = doc.Application.Create.NewLine(point05, point06, true);
            Line line06 = doc.Application.Create.NewLine(point06, point07, true);
            Line line07 = doc.Application.Create.NewLine(point07, point08, true);
            Line line08 = doc.Application.Create.NewLine(point08, point05, true);

            // Create an Array to hold the 2nd closed-loop of Profile Edges
            CurveArray curveAr2 = new CurveArray();

            // Place the single closed-loop of Profile Edges in the Array
            curveAr2.Append(line05);
            curveAr2.Append(line06);
            curveAr2.Append(line07);
            curveAr2.Append(line08);

            // Create an Array of Arrays - allows for more than one closed-loop to be used
            CurveArrArray curveArAr = new CurveArrArray();

            // Place all the closed-loops in the Array of Arrays - two in this case
            curveArAr.Append(curveAr1);
            curveArAr.Append(curveAr2);

            // Create the Extrusion - 'true' indicates it's Solid
            Extrusion extrude = doc.FamilyCreate.NewExtrusion(true, curveArAr, planeSK, height);
        }
    }
}