﻿#region Imported Namespaces

//.NET common used namespaces
using System;
using System.Linq;
using System.Collections.Generic;
using winf = System.Windows.Forms;
using wind = System.Drawing;

//Revit.NET common used namespaces
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using System.Diagnostics;
using Autodesk.Revit.DB.Architecture;

#endregion

namespace RevitNETAddin1
{
    [Transaction(TransactionMode.Manual)]
    [Regeneration(RegenerationOption.Manual)]
    public class Commands : IExternalCommand
    {

        #region class variables

        private WindowWrapper winWrap;
        private bool CanShowForm;
        private winf.Form currForm;
        private winf.ContextMenu currContextMenu;
        private UIApplication revitApp;

        #endregion

        #region Windows controls setup

        private void SetupWindowsForm(winf.Form frm)
        {
            winf.DataGridView grid = new winf.DataGridView();

            frm.Text = "Project doors";
            frm.Width = 400;
            frm.Height = 500;
            grid.Width = frm.Width - 1;
            grid.Height = frm.Height - 1;
            grid.MouseUp += new winf.MouseEventHandler(grid_MouseUp);

            SetupDataGridView(grid);
            frm.Controls.Add(grid);
            winWrap = new WindowWrapper();

            // add context menu
            currContextMenu = new winf.ContextMenu();
            winf.MenuItem mnuItemDelete = new winf.MenuItem();
            mnuItemDelete.Text = "&Delete door";
            currContextMenu.MenuItems.Add(mnuItemDelete);

            mnuItemDelete.Click += new EventHandler(mnuItemDelete_Click);
            frm.Load += new EventHandler(frm_Load);

            CanShowForm = true;
        }

        private void SetupDataGridView(winf.DataGridView grid)
        {
            grid.ColumnCount = 1;

            grid.ColumnHeadersDefaultCellStyle.BackColor = wind.Color.Navy;
            grid.ColumnHeadersDefaultCellStyle.ForeColor = wind.Color.White;
            grid.ColumnHeadersDefaultCellStyle.Font =
                new wind.Font(grid.Font, wind.FontStyle.Bold);

            grid.Name = "grid";
            grid.Location = new wind.Point(8, 8);
            grid.Size = new wind.Size(500,300);
            grid.AutoSizeRowsMode = winf.DataGridViewAutoSizeRowsMode.AllCellsExceptHeaders;
            grid.ColumnHeadersBorderStyle = winf.DataGridViewHeaderBorderStyle.Single;
            grid.CellBorderStyle = winf.DataGridViewCellBorderStyle.Single;
            grid.GridColor = wind.Color.Black;
            grid.RowHeadersVisible = false;

            winf.DataGridViewTextBoxColumn col1 = new winf.DataGridViewTextBoxColumn(); col1.Name = "Door Id";
            winf.DataGridViewTextBoxColumn col2 = new winf.DataGridViewTextBoxColumn(); col2.Name = "Door name";
            winf.DataGridViewImageColumn col3 = new winf.DataGridViewImageColumn();     col3.Name = "Door preview";

            grid.Columns.Add(col1); 
            grid.Columns.Add(col2); 
            grid.Columns.Add(col3);
            grid.Columns[0].Visible = false;

            grid.SelectionMode = winf.DataGridViewSelectionMode.FullRowSelect;
            grid.MultiSelect = false;
            grid.AllowUserToAddRows = false;
            grid.Dock = winf.DockStyle.Fill;

        }

        #endregion

        #region Methods which interacts with model

        private void PopulateDataGridView(winf.DataGridView grid)
        {
            // clear grid
            if (grid.Rows.Count > 0)
                grid.Rows.Clear();

            ElementClassFilter familyInstanceFilter = new ElementClassFilter(typeof(FamilyInstance));

            ElementCategoryFilter doorsCategoryfilter = new ElementCategoryFilter(BuiltInCategory.OST_Doors);

            LogicalAndFilter doorInstancesFilter = new LogicalAndFilter(familyInstanceFilter, doorsCategoryfilter);

            FilteredElementCollector collector = new FilteredElementCollector(revitApp.ActiveUIDocument.Document);
            IList<Element> doors = collector.WherePasses(doorInstancesFilter).ToElements();

            foreach (Element el in doors)
            {
                ElementId typeId = el.GetTypeId();
                ElementType type = (ElementType) revitApp.ActiveUIDocument.Document.get_Element(typeId);
                wind.Bitmap bmp = type.GetPreviewImage(new wind.Size(100, 100));
                Object[] row = {null,
                                el.Id.ToString(),
                                el.Name.ToString(),
                                bmp
                               };
                if (el.Name.ToString() != "Window_Insert")
                    grid.Rows.Add(row);
            }
        }

        private void DeleteDoorFromModel()
        {
            // get selected door element ID
            winf.DataGridView grd = (winf.DataGridView)currForm.Controls[0];
            int doorId = int.Parse(grd.SelectedRows[0].Cells["Door Id"].Value.ToString());
            ElementId elId = new ElementId(doorId);

            // start transaction and delete selected door from document
            string tran_Id = System.Guid.NewGuid().ToString();
            Transaction trans = new Transaction(revitApp.ActiveUIDocument.Document, tran_Id);
            trans.Start();
            revitApp.ActiveUIDocument.Document.Delete(elId);
            trans.Commit();

            revitApp.ActiveUIDocument.RefreshActiveView();

            // refresh our grid also
            grd.Rows.Remove(grd.SelectedRows[0]);
        }

        #endregion

        #region Revit and our form events

        void app_Idling(object sender, Autodesk.Revit.UI.Events.IdlingEventArgs e)
        {
            if (currForm != null)
            {
                if (!currForm.Visible)
                {
                    if (CanShowForm)
                    {
                        CanShowForm = false;
                        currForm.ShowDialog(winWrap);
                    }
                    else
                    {
                        currForm = null;
                    }
                }

                // perform message pump to our form
                winf.Application.DoEvents();
            }
        }

        void frm_Load(object sender, EventArgs e)
        {
            winf.DataGridView grid = (winf.DataGridView)currForm.Controls[0];
            PopulateDataGridView(grid);
        }

        void grid_MouseUp(object sender, winf.MouseEventArgs e)
        {
            if (e.Button == winf.MouseButtons.Right)
            {
                // get selected door element ID
                winf.DataGridView grd = (winf.DataGridView)currForm.Controls[0];
                int doorId = int.Parse(grd.SelectedRows[0].Cells["Door Id"].Value.ToString());
                ElementId elId = new ElementId(doorId);
                Element element = revitApp.ActiveUIDocument.Document.get_Element(elId);

                revitApp.ActiveUIDocument.Selection.Elements.Clear();
                revitApp.ActiveUIDocument.Selection.Elements.Add(element);
                revitApp.ActiveUIDocument.RefreshActiveView();

                currContextMenu.Show(currForm, new System.Drawing.Point(e.X, e.Y));
            }
        }

        void mnuItemDelete_Click(Object sender, EventArgs e)
        {
            DeleteDoorFromModel();
        }

        #endregion

        #region Main access point of command

        public Result Execute(ExternalCommandData commandData, ref string message, ElementSet elements)
        {
            // register for windows message pump to our form
            revitApp = commandData.Application;
            revitApp.Idling += new EventHandler<Autodesk.Revit.UI.Events.IdlingEventArgs>(app_Idling);

            // initialize our modal form
            currForm = new winf.Form();
            SetupWindowsForm(currForm);

            // return success
            return Result.Succeeded;
        }

        #endregion
    }
}
