#region Namespaces
using System;
using System.Text;
using System.Linq;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
#endregion

namespace MepSystemTraversal
{
  [Transaction( TransactionMode.ReadOnly )]
  public class Command : IExternalCommand
  {
    static string PluralSuffix( int n )
    {
      return 1 == n ? "" : "s";
    }

    void TraverseSystems( Document doc )
    {
      FilteredElementCollector systems
        = new FilteredElementCollector( doc )
          .OfClass( typeof( MEPSystem ) );

      int i, n;
      string s;
      string[] a;

      StringBuilder message = new StringBuilder();

      foreach( MEPSystem system in systems )
      {
        message.AppendLine( "System Name: "
          + system.Name );

        message.AppendLine( "Base Equipment: "
          + system.BaseEquipment );

        ConnectorSet cs = system.ConnectorManager
          .Connectors;

        i = 0;
        n = cs.Size;
        a = new string[n];

        s = string.Format(
          "{0} element{1} in ConnectorManager: ",
          n, PluralSuffix( n ) );

        foreach( Connector c in cs )
        {
          Element e = c.Owner;

          if( null != e )
          {
            a[i++] = e.GetType().Name
              + " " + e.Id.ToString();
          }
        }

        message.AppendLine( s
          + string.Join( ", ", a ) );

        i = 0;
        n = system.Elements.Size;
        a = new string[n];

        s = string.Format(
          "{0} element{1} in System: ",
          n, PluralSuffix( n ) );

        foreach( Element e in system.Elements )
        {
          a[i++] = e.GetType().Name
            + " " + e.Id.ToString();
        }

        message.AppendLine( s
          + string.Join( ", ", a ) );
      }

      n = systems.Count<Element>();

      string caption =
        string.Format( "Traverse {0} MEP System{1}",
        n, (1 == n ? "" : "s") );

      TaskDialog dlg = new TaskDialog( caption );
      dlg.MainContent = message.ToString();
      dlg.Show();
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;
      Document doc = uidoc.Document;

      TraverseSystems( doc );

      return Result.Succeeded;
    }
  }
}
