﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using Autodesk;
using Autodesk.Revit;

namespace ObjRel
{
  public partial class Result : Form
  {
    ElementSet _elems;
    ElementSet _displayedElems;
    Autodesk.Revit.Application _app;
    TreeNode rootNode;

    /// <summary>
    /// Obtain dependency information of the given BIM element
    /// by creating a transaction, using one of the overloads of the Delete method and
    /// aborting the transaction to get the dependent element Ids. For more details
    /// refer to http://thebuildingcoder.typepad.com/blog/2009/04/tag-association.html
    /// </summary>
    ElementIdSet GetDependentsElementIds( 
      Element e )
    {
      Document doc = _app.ActiveDocument;
      
      doc.BeginTransaction();

      ElementIdSet ids = doc.Delete( e );
      
      doc.AbortTransaction();

      return ids;
    }

    /// <summary>
    /// This recursive method creates the tree nodes 
    /// based on dependent element ids. Each node 
    /// displays its element name, category and id.
    /// </summary>
    void DisplayNode( Element e, TreeNode node )
    {
      string cat = (null == e.Category) 
        ? "<category unknown>" 
        : e.Category.Name;

      string label = string.Format( "{0}: {1} {2}", 
        e.Name, cat, e.Id.Value );

      TreeNode father = node.Nodes.Add( label );

      // save element id to handle select event, 
      // cf. treeView1_AfterSelect:
      
      father.ImageKey = e.Id.Value.ToString(); 

      ElementIdSet ids = GetDependentsElementIds( e );

      try
      {
        if( null != ids && 1 < ids.Size )
        {
          Document doc = _app.ActiveDocument;

          foreach( ElementId id1 in ids )
          {
            ElementId id = id1;
            Element e2 = doc.get_Element( ref id );
            if( e2 != null )
            {
              if( !e2.Id.Equals( e.Id ) 
                && !_displayedElems.Contains( e2 ) )
              {
                _displayedElems.Insert( e2 );
                DisplayNode( e2, father );
              }
            }
          }
        }
      }
      catch( Exception ex )
      {
        MessageBox.Show( ex.Message.ToString() );
      }
    }

    /// <summary>
    /// Creates the tree node with the initial node,
    /// populates it, and maintains a list of elements 
    /// already included in the nodes to avoid duplication.
    /// </summary>
    void CreateRelationships()
    {
      rootNode = new TreeNode( _app.ActiveDocument.Title );
      this.treeView1.Nodes.Add( rootNode );

      _displayedElems = new ElementSet();

      foreach( Element e in _elems )
      {
        if( !_displayedElems.Contains( e ) )
        {
          DisplayNode( e, rootNode );
        }
      }
    }

    /// <summary>
    /// Initialise the dialog with all the valid values and parameters
    /// and register the event handler on the treeview control.
    /// </summary>
    public Result( ElementSet elems, Autodesk.Revit.Application app )
    {
      _elems = elems;
      _app = app;

      InitializeComponent();
      CreateRelationships();

      treeView1.AfterSelect += new TreeViewEventHandler( treeView1_AfterSelect );
    }

    /// <summary>
    /// Handles the selection of tree node by the end user. 
    /// On selecting a particular node, this methods zooms to 
    /// the element represented by the node in the model to 
    /// show a visual cue to help identify it.
    /// </summary>
    void treeView1_AfterSelect( object sender, TreeViewEventArgs e )
    {
      if( e.Node.ImageKey.Length > 0 )
      {
        int intId = Convert.ToInt32( e.Node.ImageKey );
        ElementId id = new ElementId();
        id.Value = intId;
        Element e2 = _app.ActiveDocument.get_Element( ref id );

        _app.ActiveDocument.ShowElements( e2 );
      }
    }

    /// <summary>
    /// Closes the dialog.
    /// </summary>
    private void btnClose_Click( object sender, EventArgs e )
    {
      Close();
    }
  }
}
