#Region "Imported Namespaces"
Imports System
Imports System.Collections.Generic
Imports Autodesk.Revit.ApplicationServices
Imports Autodesk.Revit.Attributes
Imports Autodesk.Revit.DB
Imports Autodesk.Revit.UI
Imports Autodesk.Revit.UI.Selection
#End Region

<Transaction(TransactionMode.Manual)>
Public Class AdskCommand
  Implements IExternalCommand

  Const DisplayName As String = "Extract Edge"

  Public Function Execute(
    ByVal commandData As ExternalCommandData,
    ByRef message As String,
    ByVal elements As ElementSet) _
  As Result Implements IExternalCommand.Execute

    Dim uiapp As UIApplication = commandData.Application
    Dim uidoc As UIDocument = uiapp.ActiveUIDocument
    Dim doc As Document = uidoc.Document
    Dim sel As Selection = uidoc.Selection

    Dim selectedReference As Reference = sel.PickObject(
      ObjectType.Edge, "Select Edge")

    Dim selectedElement As Element = doc.GetElement(
      selectedReference.ElementId)

    Dim selectedEdge As Edge = TryCast(
      selectedElement.GetGeometryObjectFromReference(
        selectedReference), Edge)

    Dim curve As Curve = selectedEdge.AsCurve

    Dim fi As FamilyInstance = TryCast(
      selectedElement, FamilyInstance)

    If fi IsNot Nothing Then
      curve = curve.CreateTransformed(fi.GetTransform)
    End If

    Dim normal As XYZ = GetNormal(curve)

    If normal IsNot Nothing Then

      Using rvtTransaction As New Transaction(doc)
        rvtTransaction.Start(DisplayName)

        Dim plane As Plane _
          = doc.Application.Create.NewPlane(
            normal, curve.GetEndPoint(0))

        Dim sketchPlane As SketchPlane _
          = sketchPlane.Create(doc, plane)

        doc.Create.NewModelCurve(curve, sketchPlane)

        rvtTransaction.Commit()
      End Using

      Return Result.Succeeded
    Else

      MsgBox("Edge is not Planar, Cannot create model line",
             MsgBoxStyle.OkOnly, "Sorry")

      Return Result.Cancelled
    End If

  End Function

  Function GetNormal(ByVal curve As Curve) As XYZ

    Dim startpoint As XYZ = curve.GetEndPoint(0)
    Dim endpoint As XYZ = curve.GetEndPoint(1)

    If curve.GetType() = GetType(Arc) Then
      Return TryCast(curve, Arc).Normal

    ElseIf curve.GetType() = GetType(Ellipse) Then
      Return TryCast(curve, Ellipse).Normal

    ElseIf curve.GetType() = GetType(Line) Then
      Return startpoint.CrossProduct(endpoint) _
        .Normalize()

    Else

      Dim pointList As IList(Of XYZ) _
        = curve.Tessellate()

      Dim vectorToEndpoint As XYZ _
        = pointList(pointList.Count - 1) _
          - pointList(0)

      Dim lastVector As XYZ = Nothing
      Dim firstNormal As XYZ = Nothing
      Dim nextNormal As XYZ = Nothing

      For i As Integer = 1 To pointList.Count - 2

        Dim nextPoint As XYZ = pointList(i)

        Dim nextVector As XYZ = nextPoint.Subtract(
          startpoint)

        If firstNormal Is Nothing Then
          firstNormal = AbsoluteVector(
            vectorToEndpoint.CrossProduct(
              nextVector).Normalize)
        Else
          nextNormal = AbsoluteVector(
            vectorToEndpoint.CrossProduct(
              nextVector).Normalize)
          If Not IsZero(firstNormal.DistanceTo(
                        nextNormal)) Then
            Return Nothing
          End If
        End If

        lastVector = nextVector
      Next i

      Return vectorToEndpoint.CrossProduct(lastVector) _
        .Normalize()

    End If

  End Function

  Function AbsoluteVector(ByVal vector As XYZ) As XYZ
    Return New XYZ(System.Math.Abs(vector.X),
                   System.Math.Abs(vector.Y),
                   System.Math.Abs(vector.Z))
  End Function

  Public Function IsZero(ByVal number As Double) _
    As Boolean
    Return 0.000000001 > System.Math.Abs(number)
  End Function

End Class





