#region Namespaces
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.Plumbing;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
using RvtOperationCanceledException = Autodesk.Revit.Exceptions.OperationCanceledException;
#endregion

namespace PipeInsulationMaterial
{
  [Transaction( TransactionMode.ReadOnly )]
  public class Command : IExternalCommand
  {
    const string _caption = "Pipe Insulation Info";

    /// <summary>
    /// Restrict user selection to pipe element only
    /// </summary>
    class PipeFilter : ISelectionFilter
    {
      public bool AllowElement( Element e )
      {
        return e is Pipe;
      }

      public bool AllowReference( Reference r, XYZ p )
      {
        return true;
      }
    }

    /// <summary>
    /// Return first pre-selected element or 
    /// prompt user to select a pipe.
    /// </summary>
    Element GetSelectedElement( UIDocument uidoc )
    {
      Element e = null;

      SelElementSet elements = uidoc.Selection.Elements;

      if( null == elements || elements.IsEmpty )
      {
        try
        {
          Reference r = uidoc.Selection.PickObject(
            ObjectType.Element,
            new PipeFilter(),
            "Please pick a pipe" );

          e = uidoc.Document.GetElement( r.ElementId );
        }
        catch( RvtOperationCanceledException )
        {
        }
      }
      else if( null != elements && !elements.IsEmpty )
      {
        IEnumerator it = elements.GetEnumerator();

        if( it.MoveNext() )
        {
          e = it.Current as Element;
        }
      }
      return e;
    }

    /// <summary>
    /// Return pipe insulation for given pipe using 
    /// filtered element collector and HostElementId
    /// property or InsulationLiningBase 
    /// GetInsulationIds method.
    /// </summary>
    PipeInsulation GetPipeInslationFromPipe(
      Pipe pipe )
    {
      if( pipe == null )
      {
        throw new ArgumentNullException( "pipe" );
      }

      Document doc = pipe.Document;

      // Filtered element collector and HostElementId

      FilteredElementCollector fec
        = new FilteredElementCollector( doc )
          .OfClass( typeof( PipeInsulation ) );

      PipeInsulation pipeInsulation = null;

      foreach( PipeInsulation pi in fec )
      {
        // Find the first insulation
        // belonging to the given pipe

        if( pi.HostElementId == pipe.Id )
        {
          pipeInsulation = pi;
          break;
        }
      }

#if DEBUG
      // InsulationLiningBase.GetInsulationIds method
      // returns all pipe insulations for a given pipe

      ICollection<ElementId> pipeInsulationIds 
        = InsulationLiningBase.GetInsulationIds( 
          doc, pipe.Id );

      Debug.Assert( 
        pipeInsulationIds.Contains( pipeInsulation.Id ),
        "expected InsulationLiningBase.GetInsulationIds"
        + " to include pipe insulation element id" );
#endif // DEBUG

      return pipeInsulation;
    }

    /// <summary>
    /// Return material from given pipe insulation.
    /// </summary>
    Material GetMaterialFromPipeInsulation(
      PipeInsulation pipeInsulation )
    {
      if( pipeInsulation == null )
      {
        throw new ArgumentNullException( "pipeInsulation" );
      }

      Document doc = pipeInsulation.Document;

      PipeInsulationType pipeInsulationType
        = doc.GetElement( pipeInsulation.GetTypeId() )
          as PipeInsulationType;

      Parameter p = pipeInsulationType.get_Parameter(
        "Material" );

      return null == p
        ? null
        : doc.GetElement( p.AsElementId() ) as Material;
    }

    /// <summary>
    /// Prompt user to select a pipe, retrieve its 
    /// insulation and insulation material, 
    /// and report the results.
    /// </summary>
    void GetInsulationFromSelection(
      UIDocument uidoc )
    {
      Element e = GetSelectedElement( uidoc );

      if( null != e )
      {
        if( e is Pipe )
        {
          PipeInsulation pi
            = GetPipeInslationFromPipe( e as Pipe );

          if( null == pi )
          {
            TaskDialog.Show( _caption,
              "Insulation not found" );
          }
          else
          {
            Material material
              = GetMaterialFromPipeInsulation( pi );

            if( null == material )
            {
              TaskDialog.Show( _caption,
                "Material not found" );
            }
            else
            {
              TaskDialog.Show( _caption,
                string.Format( "Material '{0}' <{1}>",
                material.Name, material.Id ) );
            }
          }
        }
        else
        {
          TaskDialog.Show( _caption, "Not a pipe" );
        }
      }
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;

      GetInsulationFromSelection( uidoc );

      return Result.Succeeded;
    }
  }
}
