#region Namespaces
using System;
using System.Collections.Generic;
using Autodesk.Revit.ApplicationServices;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;
#endregion

namespace PreventDeletion
{
  /// <summary>
  /// External command to select elements 
  /// to protect from deletion.
  /// </summary>
  [Transaction( TransactionMode.ReadOnly )]
  public class Command : IExternalCommand
  {
    public const string Caption = "Prevent Deletion";

    static List<ElementId> _protectedIds 
      = new List<ElementId>();

    static public bool IsProtected( ElementId id )
    {
      return _protectedIds.Contains( id );
    }

    public Result Execute(
      ExternalCommandData commandData,
      ref string message,
      ElementSet elements )
    {
      UIApplication uiapp = commandData.Application;
      UIDocument uidoc = uiapp.ActiveUIDocument;

      IList<Reference> refs = null;

      try
      {
        Selection sel = uidoc.Selection;

        refs = sel.PickObjects( ObjectType.Element, 
          "Please pick elements to prevent from deletion. " );
      }
      catch( OperationCanceledException )
      {
        return Result.Cancelled;
      }

      if( null != refs && 0 < refs.Count )
      {
        foreach( Reference r in refs )
        {
          ElementId id = r.ElementId;

          if( !_protectedIds.Contains( id ) )
          {
            _protectedIds.Add( id );
          }
        }
        int n = refs.Count;

        TaskDialog.Show( Caption, string.Format(
          "{0} new element{1} selected and protected "
          + " from deletion, {2} in total.",
          n, ( 1 == n ? "" : "s" ), 
          _protectedIds.Count ) );
      }
      return Result.Succeeded;
    }
  }
}
