﻿using System;
using System.IO;
using System.IO.Packaging;
using System.Reflection;
using System.Runtime.InteropServices;

namespace ConsoleApplication1
{
  class Program
  {
    private const string StreamName = "BasicFileInfo";

    static void Main( string[] args )
    {
      string pathToRevitFile = ( 1 == args.Length )
        ? args[0]
        : @"pathToRevitFile";
     

      if( !StructuredStorageUtils.IsFileStucturedStorage(
        pathToRevitFile ) )
      {
        throw new NotSupportedException(
          "File is not a structured storage file" );
      }

      var rawData = GetRawBasicFileInfo( 
        pathToRevitFile );

      var rawString = System.Text.Encoding.Unicode
        .GetString( rawData );

      var fileInfoData = rawString.Split(
        new string[] { "\0", "\r\n" },
        StringSplitOptions.RemoveEmptyEntries );

      foreach( var info in fileInfoData )
      {
        Console.WriteLine( info );
      }

      Console.WriteLine( "Big-endian:" );

      rawString = System.Text.Encoding.BigEndianUnicode
        .GetString( rawData );

      fileInfoData = rawString.Split(
        new string[] { "\0", "\r\n" },
        StringSplitOptions.RemoveEmptyEntries );

      foreach( var info in fileInfoData )
      {
        Console.WriteLine( info );
      }
    }

    private static byte[] GetRawBasicFileInfo(
      string revitFileName )
    {
      if( !StructuredStorageUtils.IsFileStucturedStorage(
        revitFileName ) )
      {
        throw new NotSupportedException(
          "File is not a structured storage file" );
      }

      using( StructuredStorageRoot ssRoot =
          new StructuredStorageRoot( revitFileName ) )
      {
        if( !ssRoot.BaseRoot.StreamExists( StreamName ) )
          throw new NotSupportedException( string.Format(
            "File doesn't contain {0} stream", StreamName ) );

        StreamInfo imageStreamInfo =
            ssRoot.BaseRoot.GetStreamInfo( StreamName );

        using( Stream stream = imageStreamInfo.GetStream(
          FileMode.Open, FileAccess.Read ) )
        {
          byte[] buffer = new byte[stream.Length];
          stream.Read( buffer, 0, buffer.Length );
          return buffer;
        }
      }
    }
  }

  public static class StructuredStorageUtils
  {
    [DllImport( "ole32.dll" )]
    static extern int StgIsStorageFile(
      [MarshalAs( UnmanagedType.LPWStr )]
      string pwcsName );

    public static bool IsFileStucturedStorage(
      string fileName )
    {
      int res = StgIsStorageFile( fileName );

      if( res == 0 )
        return true;

      if( res == 1 )
        return false;

      throw new FileNotFoundException(
        "File not found", fileName );
    }
  }

  public class StructuredStorageException : Exception
  {
    public StructuredStorageException()
    {
    }

    public StructuredStorageException( string message )
      : base( message )
    {
    }

    public StructuredStorageException(
      string message,
      Exception innerException )
      : base( message, innerException )
    {
    }
  }

  public class StructuredStorageRoot : IDisposable
  {
    StorageInfo _storageRoot;

    public StructuredStorageRoot( Stream stream )
    {
      try
      {
        _storageRoot
          = (StorageInfo)InvokeStorageRootMethod(
            null, "CreateOnStream", stream );
      }
      catch( Exception ex )
      {
        throw new StructuredStorageException(
          "Cannot get StructuredStorageRoot", ex );
      }
    }

    public StructuredStorageRoot( string fileName )
    {
      try
      {
        _storageRoot
          = (StorageInfo)InvokeStorageRootMethod(
            null, "Open", fileName, FileMode.Open,
            FileAccess.Read, FileShare.Read );
      }
      catch( Exception ex )
      {
        throw new StructuredStorageException(
          "Cannot get StructuredStorageRoot", ex );
      }
    }

    private static object InvokeStorageRootMethod(
      StorageInfo storageRoot,
      string methodName,
      params object[] methodArgs )
    {
      Type storageRootType
        = typeof( StorageInfo ).Assembly.GetType(
          "System.IO.Packaging.StorageRoot",
          true, false );

      object result = storageRootType.InvokeMember(
        methodName,
        BindingFlags.Static | BindingFlags.Instance
        | BindingFlags.Public | BindingFlags.NonPublic
        | BindingFlags.InvokeMethod,
        null, storageRoot, methodArgs );

      return result;
    }

    private void CloseStorageRoot()
    {
      InvokeStorageRootMethod( _storageRoot, "Close" );
    }

    #region Implementation of IDisposable

    public void Dispose()
    {
      CloseStorageRoot();
    }

    #endregion

    public StorageInfo BaseRoot
    {
      get { return _storageRoot; }
    }

    //private void ControlledApplication_DocumentOpening( 
    //  object sender,
    //  DocumentOpeningEventArgs e )
    //{
    //  FileInfo revitFileToUpgrade 
    //    = new FileInfo( e.PathName );

    //  Regex buildInfoRegex = new Regex(
    //    @"Revit\sArchitecture\s(?<Year>\d{4})\s"
    //    +@"\(Build:\s(?<Build>\w*)\((<Processor>\w{3})\)\)" );

    //  using( StreamReader streamReader =
    //    new StreamReader( e.PathName, Encoding.Unicode ) )
    //  {
    //    string fileContents = streamReader.ReadToEnd();

    //    Match buildInfo = buildInfoRegex.Match( fileContents );
    //    string year = buildInfo.Groups["Year"].Value;
    //    string build = buildInfo.Groups["Build"].Value;
    //    string processor = buildInfo.Groups["Processor"].Value;
    //  }
    //}

  }
}
