﻿using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;

using ObjExporter.Model;

namespace ObjExporter
{
    [Transaction(TransactionMode.Manual)]
    [Regeneration(RegenerationOption.Manual)]
    public class Controller : IExternalCommand
    {
        public Result Execute(ExternalCommandData commandData, ref string message, ElementSet elements)
        {
            try
            {
                //get the active revit document
                UIDocument uidoc = commandData.Application.ActiveUIDocument;
                Document doc = uidoc.Document;

                //check if the active view is a 3D view
                if (uidoc.ActiveView as View3D == null)
                {
                    TaskDialog.Show("Message", "Start this tool from a 3D view");
                    return Result.Succeeded;
                }

                string REVIT_MODEL_EXPORT_PATH = "";

                #region ask the user the specify a file location
                //first set a defaulft path based on the projectname and the view name
                string defaultFileName = string.Format("ProjectName - {0}.obj", uidoc.ActiveView.Name);
                if(doc.PathName != null)
                {
                    if(doc.PathName.Length > 0)
                    {
                        defaultFileName = string.Format(@"{0} - {1}.obj", System.IO.Path.GetFileNameWithoutExtension(doc.PathName), uidoc.ActiveView.Name);
                    }
                }
             
                //then show a save file dialog with this default file name
                using (System.Windows.Forms.SaveFileDialog sfd = new System.Windows.Forms.SaveFileDialog())
                {
                    sfd.Filter = "obj file|*.obj";
                    sfd.Title = "Save an obj file";
                    sfd.FileName = defaultFileName;
                    sfd.DefaultExt = "obj";
                    sfd.AddExtension = true;
                    if(sfd.ShowDialog() != System.Windows.Forms.DialogResult.OK)
                    {
                        return Result.Succeeded;
                    }
                    else
                    {
                        REVIT_MODEL_EXPORT_PATH = sfd.FileName;
                    }
                }
                #endregion

                // Instantiate our custom context  
                ExportOptions options = new ExportOptions();
                options.IncludeFaces = false;
                options.IncludeMeshes = false;
                ExportContext context = new ExportContext(doc, REVIT_MODEL_EXPORT_PATH, options);

                //Get the active view to export
                IList<ElementId> viewIdsToExport = new List<ElementId>() { uidoc.ActiveView.Id }; 

                // export
                using (CustomExporter exporter = new CustomExporter(doc, context))
                {
                    // Tell the exportrd whether we need face info.
                    // If not, it is better to exclude them, since 
                    // processing faces takes significant time and 
                    // memory. In any case, tessellated polymeshes
                    // can be exported (and will be sent to the 
                    // context).

                    exporter.IncludeGeometricObjects = options.IncludeFaces;
                    

                    try
                    {
                        exporter.Export(viewIdsToExport);
                    }
                    catch (System.Exception ex)
                    {
                        string msg = "Exception: " + (ex.Message == null ? "No message included": ex.Message);
                        TaskDialog.Show("Error", msg);
                        return Result.Failed;
                    }
                }
            }
            catch(Exception ex)
            {
                return Result.Succeeded;
            }
            return Result.Succeeded;
        }

        
    }

   
}
