﻿using System;
using System.IO;
using System.Text;
using System.Xml;
using System.Collections.Generic;
using System.Diagnostics;

using Autodesk.Revit.DB.Visual;
using Autodesk.Revit.DB;

namespace ObjExporter.Model
{
    /// <summary>
    /// A utility class to handle exporting (dumping) of assets
    /// </summary>
    class ExportAsset
    {
        /// <summary>
        /// This writes an entire asset structure with all its items and substructures
        /// </summary>
        internal static void WriteAssetElement(XmlWriter writer, Asset asset)
        {
            if (asset != null)
            {
                string tag = asset.Name.Replace(' ', '_'); // jeremy
                                                           //Debug.Assert( !tag.Contains( " " ), "invalid char in XML tag " + tag ); // jeremy
                writer.WriteStartElement(tag);
                if (!string.IsNullOrEmpty(asset.Title))
                {
                    writer.WriteAttributeString("Title", asset.Title);
                }
                writer.WriteAttributeString("AssetType", asset.AssetType.ToString());
                writer.WriteAttributeString("Library", asset.LibraryName);
                WriteAssetProperties(writer, asset);
                writer.WriteEndElement();
            }
        }

        /// <summary>
        /// This writes a collections of asset properties  (basically, an asset represents such a collection)
        /// </summary>
        private static void WriteAssetProperties(XmlWriter writer, AssetProperties properties)
        {
            if (properties != null)
            {
                for (int index = 0; index < properties.Size; index++)
                {
                    WriteAssetProperty(writer, properties[index]);
                }
            }
        }


        /// <summary>
        /// This writes one property as a pair of name and value
        /// </summary>
        /// <remarks>
        /// It is written either as a single-line element or a compound element
        /// depending on the request whether the property's elements should be left open.
        /// If it is to be left open it is probably more will be added to it (such as connected properties)
        /// </remarks>
        private static void WriteSingleProperty(XmlWriter writer, String name, String value, bool leaveOpen)
        {
            string tag = name.Replace(' ', '_'); // jeremy
                                                 //Debug.Assert( !tag.Contains( " " ), "invalid char in XML tag " + tag ); // jeremy
            if (leaveOpen)
            {
                writer.WriteStartElement(tag);
                writer.WriteAttributeString("value", value);
            }
            else
            {
                writer.WriteElementString(tag, value);
            }
        }


        /// <summary>
        /// This writes all properties connected to the given property
        /// </summary>
        /// <remarks>
        /// A connected property is an asset structure that can be
        /// attached to another asset structure and modify (or extend)
        /// its definition.
        /// </remarks>
        private static void WriteConnectedProperties(XmlWriter writer, AssetProperty prop)
        {
            if (prop.NumberOfConnectedProperties > 0)
            {
                if (1 == prop.NumberOfConnectedProperties)
                {
                    WriteAssetProperty(writer, prop.GetConnectedProperty(0));
                }
                else
                {
                    writer.WriteAttributeString("Count", prop.NumberOfConnectedProperties.ToString());

                    for (int idx = 0; idx < prop.NumberOfConnectedProperties; idx++)
                    {
                        WriteAssetProperty(writer, prop.GetConnectedProperty(idx));
                    }

                }
            }
        }



        /// <summary>
        /// This writes one asset property in a form that depends on the property type
        /// </summary>
        private static void WriteAssetProperty(XmlWriter writer, AssetProperty prop)
        {
            if (prop != null)
            {
                // we need to know if there will be any connected properties added
                // to this property. If there are, we will add them at the end of this method.
                bool hasConnectedProperties = (prop.NumberOfConnectedProperties > 0);

                // if we have connected properties we always leave the element open
                // because it will be closed after connected properties are also written,
                // but we leave compound properties always open for simplicity,
                // so compound properties will always need to be closed here at the end.
                bool needsEndElementTag = hasConnectedProperties;

                switch (prop.Type)
                {
                    case AssetPropertyType.Boolean:
                        {
                            AssetPropertyBoolean e = prop as AssetPropertyBoolean;
                            WriteSingleProperty(writer, prop.Name, e.Value.ToString(), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Enumeration:
                        {
                            AssetPropertyEnum e = prop as AssetPropertyEnum;
                            WriteSingleProperty(writer, prop.Name, e.Value.ToString(), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Integer:
                        {
                            AssetPropertyInteger e = prop as AssetPropertyInteger;
                            WriteSingleProperty(writer, prop.Name, e.Value.ToString(), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Longlong:
                        {
                            AssetPropertyInt64 e = prop as AssetPropertyInt64;
                            WriteSingleProperty(writer, prop.Name, e.Value.ToString(), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.ULonglong:
                        {
                            AssetPropertyUInt64 e = prop as AssetPropertyUInt64;
                            WriteSingleProperty(writer, prop.Name, e.Value.ToString(), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Float:
                        {
                            AssetPropertyFloat e = prop as AssetPropertyFloat;
                            WriteSingleProperty(writer, prop.Name, e.Value.ToString(), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Double1:
                        {
                            AssetPropertyDouble e = prop as AssetPropertyDouble;
                            WriteSingleProperty(writer, prop.Name, e.Value.ToString(), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.String:
                        {
                            AssetPropertyString e = prop as AssetPropertyString;
                            WriteSingleProperty(writer, prop.Name, e.Value, hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Time:
                        {
                            AssetPropertyTime e = prop as AssetPropertyTime;
                            WriteSingleProperty(writer, prop.Name, e.Value.ToString(), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Distance:
                        {
                            AssetPropertyDistance e = prop as AssetPropertyDistance;
                            WriteSingleProperty(writer, prop.Name, e.Value.ToString(), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Double2:
                        {
                            AssetPropertyDoubleArray2d e = prop as AssetPropertyDoubleArray2d;
                            WriteSingleProperty(writer, prop.Name, StringConversions.ValueToString(e.Value), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Double3:
                        {
                            AssetPropertyDoubleArray3d e = prop as AssetPropertyDoubleArray3d;
                            WriteSingleProperty(writer, prop.Name, StringConversions.ValueToString(e.GetValueAsDoubles()), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Double4:
                        {
                            AssetPropertyDoubleArray4d e = prop as AssetPropertyDoubleArray4d;
                            WriteSingleProperty(writer, prop.Name, StringConversions.ValueToString(e.GetValueAsDoubles()), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Double44:
                        {
                            AssetPropertyDoubleMatrix44 e = prop as AssetPropertyDoubleMatrix44;
                            WriteSingleProperty(writer, prop.Name, StringConversions.ValueToString(e.Value), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.Float3:
                        {
                            AssetPropertyFloatArray e = prop as AssetPropertyFloatArray;
                            WriteSingleProperty(writer, prop.Name, StringConversions.ValueToString(e.GetValue()), hasConnectedProperties);
                            break;
                        }
                    case AssetPropertyType.List:
                        {
                            AssetPropertyList e = prop as AssetPropertyList;
                            string tag = e.Name.Replace(' ', '_'); // jeremy
                                                                   //Debug.Assert( !tag.Contains( " " ), "invalid char in XML tag " + tag ); // jeremy
                            writer.WriteStartElement(tag);
                            if (e.GetValue().Count > 0)
                            {
                                writer.WriteAttributeString("Count", e.GetValue().Count.ToString());
                                foreach (AssetProperty item in e.GetValue())
                                {
                                    WriteAssetProperty(writer, item);
                                }
                            }
                            needsEndElementTag = true;
                            break;
                        }
                    case AssetPropertyType.Reference:
                        {
                            AssetPropertyReference e = prop as AssetPropertyReference;
                            if (e != null)
                            {
                                string tag = e.Name.Replace(' ', '_'); // jeremy
                                                                       //Debug.Assert( !tag.Contains( " " ), "invalid char in XML tag " + tag ); // jeremy
                                writer.WriteStartElement(tag);
                                needsEndElementTag = true;
                            }
                            break;
                        }
                    case AssetPropertyType.Asset:
                        {
                            Asset e = prop as Asset;
                            if (e != null)
                            {
                                WriteAssetElement(writer, e);
                            }
                            break;
                        }
                    case AssetPropertyType.Properties:
                        {
                            AssetProperties e = prop as AssetProperties;
                            string tag = e.Name.Replace(' ', '_'); // jeremy
                                                                   //Debug.Assert( !tag.Contains( " " ), "invalid char in XML tag " + tag ); // jeremy
                            if (e != null)
                            {
                                writer.WriteStartElement(tag);
                                WriteAssetProperties(writer, e);
                                needsEndElementTag = true;
                            }
                            break;
                        }
                    case AssetPropertyType.Unknown:
                    default:
                        {
                            WriteSingleProperty(writer, prop.Name, prop.ToString(), hasConnectedProperties);
                            break;
                        }

                }

                // write "connected" properties if there are any
                if (hasConnectedProperties)
                {
                    WriteConnectedProperties(writer, prop);
                }

                // write the end tag if we haven't done so yet
                if (needsEndElementTag)
                {
                    writer.WriteEndElement();
                }
            }
        }
    }
}
