﻿using System;
using System.IO;
using System.Text;
using System.Collections.Generic;

using Autodesk.Revit.DB;

namespace ObjExporter.Model
{
    public class ExportUtilities
    {
    }
    /// <summary>
    /// Miscellaneous conversion (to a string) utilities
    /// </summary>
    public class StringConversions
    {
        internal static String ValueToString(Transform t)
        {
            if (t == Transform.Identity)
            {
                return "(1,0,0)(0,1,0)(0,0,1)(0,0,0)";
            }

            StringBuilder sb = new StringBuilder(150);
            sb.Append(t.BasisX.ToString());
            sb.Append(t.BasisY.ToString());
            sb.Append(t.BasisZ.ToString());
            sb.Append(t.Origin.ToString());
            return sb.ToString();
        }

        internal static String ValueToString(IList<float> values)
        {
            StringBuilder sb = new StringBuilder(15 * values.Count);
            bool first = true;
            foreach (float item in values)
            {
                if (!first)
                {
                    sb.Append(",");
                    first = false;
                }
                sb.Append(item.ToString());
            }
            return sb.ToString();
        }

        internal static String ValueToString(IList<XYZ> Points)
        {
            StringBuilder sb = new StringBuilder(45 * Points.Count);
            foreach (XYZ point in Points)
            {
                sb.Append(point.ToString());
            }
            return sb.ToString();
        }

        internal static String ValueToString(IList<UV> Points)
        {
            StringBuilder sb = new StringBuilder(30 * Points.Count);
            foreach (UV point in Points)
            {
                sb.Append(point.ToString());
            }
            return sb.ToString();
        }

        internal static String ValueToString(DoubleArray doubles)
        {
            StringBuilder sb = new StringBuilder(20 * doubles.Size);
            if (doubles.Size > 0)
            {
                sb.Append(doubles.get_Item(0).ToString());
                for (int idx = 1; idx < doubles.Size; idx++)
                {
                    sb.Append(",");
                    sb.Append(doubles.get_Item(idx).ToString());
                }
            }
            return sb.ToString();
        }
        internal static String ValueToString(IList<double> doubles)
        {
            StringBuilder sb = new StringBuilder(20 * doubles.Count);
            if (doubles.Count > 0)
            {
                sb.Append(doubles[0].ToString());
                for (int idx = 1; idx < doubles.Count; idx++)
                {
                    sb.Append(",");
                    sb.Append(doubles[idx].ToString());
                }
            }
            return sb.ToString();
        }

        internal static String ValueToString(IList<PolymeshFacet> facets)
        {
            StringBuilder sb = new StringBuilder(12 * facets.Count);
            foreach (PolymeshFacet facet in facets)
            {
                sb.Append(facet.ToString());
            }
            return sb.ToString();
        }
    }
    /// <summary>
    /// A helper class for caching stacked transformations
    /// </summary>
    public class Transformations
    {
        // A stack of output transformations.
        // The stack contains combined transformations, not individual transforms

        private Stack<Transform> m_transforms = new Stack<Transform>();
        private Transform m_currentTransform = Transform.Identity;

        // adding a transform on the stack
        public void PushTransform(Transform t)
        {
            // the current transform is added to the stack
            m_transforms.Push(m_currentTransform);
            // and then a new current transform is calculated by combining with the given one
            m_currentTransform = m_currentTransform.Multiply(t);
        }

        // removing one transform from the stack
        public void PopTransform()
        {
            if (m_transforms.Count > 0)
            {
                // the new current transform will be the one removed from top of the stack
                m_currentTransform = m_transforms.Pop();
            }
            else
            {
                // this situation would deserve a debug warning
            }
        }

        // clearing the stack
        public void ClearTransforms()
        {
            m_transforms.Clear();
            m_currentTransform = Transform.Identity;
        }

        // applying the current transform to a collection of points
        public IList<XYZ> ApplyTransform(IList<XYZ> points)
        {
            IList<XYZ> newPoints = new List<XYZ>(points.Count);

            foreach (XYZ xyz in points)
            {
                newPoints.Add(m_currentTransform.OfPoint(xyz));
            }

            return newPoints;
        }
        public XYZ ApplyTransform(XYZ point)
        {
            return m_currentTransform.OfPoint(point);
        }
    }  // class Transformations
    public class VertexLookupInt : Dictionary<PointInt, int>
    {
        #region PointIntEqualityComparer
        /// <summary>
        /// Define equality for integer-based PointInt.
        /// </summary>
        class PointIntEqualityComparer : IEqualityComparer<PointInt>
        {
            public bool Equals(PointInt p, PointInt q)
            {
                return 0 == p.CompareTo(q);
            }

            public int GetHashCode(PointInt p)
            {
                return (p.X.ToString()
                  + "," + p.Y.ToString()
                  + "," + p.Z.ToString())
                  .GetHashCode();
            }
        }
        #endregion // PointIntEqualityComparer

        public VertexLookupInt()
          : base(new PointIntEqualityComparer())
        {
        }

        /// <summary>
        /// Return the index of the given vertex,
        /// adding a new entry if required.
        /// </summary>
        public int AddVertex(PointInt p)
        {
            return ContainsKey(p)
              ? this[p]
              : this[p] = Count;
        }
    }
    public class PointInt : IComparable<PointInt>
    {
        public int X { get; set; }
        public int Y { get; set; }
        public int Z { get; set; }

        //public PointInt( int x, int y, int z )
        //{
        //  X = x;
        //  Y = y;
        //  Z = z;
        //}

        const double _feet_to_mm = 3048;

        static int ConvertFeetToMillimetres(double d)
        {
            return (int)(_feet_to_mm * d + 0.5);
        }

        public PointInt(XYZ p)
        {
            X = ConvertFeetToMillimetres(p.X);
            Y = ConvertFeetToMillimetres(p.Y);
            Z = ConvertFeetToMillimetres(p.Z);
        }

        public int CompareTo(PointInt a)
        {
            int d = X - a.X;

            if (0 == d)
            {
                d = Y - a.Y;

                if (0 == d)
                {
                    d = Z - a.Z;
                }
            }
            return d;
        }
    }


}
