﻿using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using ObjExporter.Model;

namespace RevitPlugin
{
    [Transaction(TransactionMode.Manual)]
    [Regeneration(RegenerationOption.Manual)]
    public class Controller : IExternalCommand
    {
        public Result Execute(ExternalCommandData commandData, ref string message, ElementSet elements)
        {
            //get the active revit document
            UIDocument uidoc = commandData.Application.ActiveUIDocument;
            Document doc = uidoc.Document;

            //check if the active view is a 3D view
            if(uidoc.ActiveView as View3D == null)
            {
                TaskDialog.Show("Message", "Start this tool from a 3D view");
                return Result.Succeeded;
            }

            //first check if the viewer window/process is already open. If so, activate it instead of  exporting and opening a new process. 
            //Maybe implement some kind of refresh function in the future to make sure that the viewer displays any changes made to the revit document after the viewer window was opened
            if (MessageHandler.viewer_ProcessId != 0)
            {
                // the following line could be enough, but rather activate the window thru the process
                SetForegroundWindow(MessageHandler.viewer_hwnd);

                // try to activate the existing instance
                Process[] processes = Process.GetProcesses();

                foreach (Process p in processes)
                {
                    if (p.Id == MessageHandler.viewer_ProcessId)
                    {
                        IntPtr windowHandle = p.MainWindowHandle;
                        SetForegroundWindow(windowHandle);
                        return Result.Succeeded;
                    }
                }
            }

            #region export the active view
            //Instantiate te exporter context
            string REVIT_MODEL_EXPORT_FILENAME = @"C:\ProgramData\Autodesk\Revit\Addins\2018\Revit External Model Viewer\ViewerContent.obj";
            if (!System.IO.Directory.Exists(System.IO.Path.GetDirectoryName(REVIT_MODEL_EXPORT_FILENAME)))
            {
                System.IO.Directory.CreateDirectory(System.IO.Path.GetDirectoryName(REVIT_MODEL_EXPORT_FILENAME));
            }
            ExportOptions options = new ExportOptions();
            options.IncludeFaces = false;
            options.IncludeMeshes = false;
            ExportContext context = new ExportContext(doc, REVIT_MODEL_EXPORT_FILENAME, options);

            //Get the active view to export
            IList<ElementId> viewIdsToExport = new List<ElementId>() { uidoc.ActiveView.Id };

            // Instantiate a custom exporter with output 
            // context given as the argument

            using (CustomExporter exporter = new CustomExporter(doc, context))
            {
                // Tell the exportrd whether we need face info.
                // If not, it is better to exclude them, since 
                // processing faces takes significant time and 
                // memory. In any case, tessellated polymeshes
                // can be exported (and will be sent to the 
                // context).

                exporter.IncludeGeometricObjects = options.IncludeFaces;

                try
                {
                    exporter.Export(viewIdsToExport);
                }
                catch (System.Exception ex)
                {
                    string msg = "Exception: " + ex.Message;
                    return Result.Failed;
                }
            }
            #endregion

            // execute the viewer window process
            Process process = new Process();
            process.StartInfo.FileName = @"C:\Users\Jesse-M\source\repos\Viewer development\RevitExternalModelViewer\ModelViewer\bin\ModelViewer.exe ";
            process.StartInfo.Arguments = ExternalApplication.messagehandler.Handle.ToString(); // pass the MessageHandler's window handle the the process as a command line argument
            process.Start();

            MessageHandler.viewer_ProcessId = process.Id; // grab the PID so we can kill the process if required;

            return Result.Succeeded;
        }

        [DllImport("user32.dll")]
        [return: MarshalAs(UnmanagedType.Bool)]
        static extern bool SetForegroundWindow(IntPtr hWnd);

        /// <summary>
        /// Gets the path to the folder where this files compiled code lies 
        /// </summary>
        public static string AssemblyDirectory
        {
            get { return System.IO.Path.GetDirectoryName(new System.Uri(System.Reflection.Assembly.GetExecutingAssembly().CodeBase).LocalPath); }
        }
    }

    
}
