﻿using System;
using System.IO;
using System.Collections.Generic;

using winform = System.Windows.Forms;
using Autodesk.Revit.Attributes;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Events;
using Autodesk.Revit.UI.Selection;

namespace RevitUIAutoSamples
{
  [Transaction( TransactionMode.Manual )]
  [Regeneration( RegenerationOption.Manual )]
  public class Command_ImportDWGCurves : IExternalCommand
  {
    public Result Execute( 
      ExternalCommandData revit, 
      ref String message, 
      ElementSet elements )
    {
      return Execute( revit.Application );
    }

    public Result Execute( UIApplication revit )
    {
      if( revit.ActiveUIDocument.Document.IsFamilyDocument )
      {
        winform.MessageBox.Show( "File must not be a family file." );
      }
      else
      {
        if( revit.ActiveUIDocument.Document.ActiveView.ViewType != ViewType.FloorPlan )
        {
          winform.MessageBox.Show( "Current view must be a floor plan." );
        }
        else
        {
          Transaction transaction = new Transaction( revit.ActiveUIDocument.Document );
          transaction.SetName( "Invoking (" + UIApp.invokeCounter + ")" );
          transaction.Start();
          try
          {
            if( UIApp.invokeCounter == 0 ) // do the first part of your command
            {
              // import a dwg, select the importinstance...

              string strFileName = "";

              winform.OpenFileDialog openFileDialog = new winform.OpenFileDialog();
              openFileDialog.Title = "Please select a DWG...";

              openFileDialog.InitialDirectory = Environment.GetFolderPath( Environment.SpecialFolder.Personal );

              openFileDialog.Filter = "AutoCAD Drawings" + " (*.dwg)|*.dwg";


              if( openFileDialog.ShowDialog() == winform.DialogResult.OK )
              {
                strFileName = openFileDialog.FileName;

                if( strFileName != "" )
                {
                  DWGImportOptions dio = new DWGImportOptions();

                  dio.View = revit.ActiveUIDocument.Document.ActiveView;
                  Element instance = null;

                  bool canImport = revit.ActiveUIDocument.Document.Import( strFileName, dio, out instance );

                  if( !canImport )
                  {
                    winform.MessageBox.Show( "Cannot import dwg file." );
                  }
                  else
                  {
                    SelElementSet seleset = SelElementSet.Create();

                    SelElementSet selements = revit.ActiveUIDocument.Selection.Elements;

                    selements = revit.ActiveUIDocument.Selection.Elements;
                    seleset.Insert( instance );

                    selements.Clear();

                    revit.ActiveUIDocument.Selection.Elements = seleset;

                    revit.ActiveUIDocument.Document.Regenerate();

                    UIApp.invokeCounter = 1; // only increase if all conditions are satisfied
                  }
                }
              }
            }
            else // do the second part of your command
            {
              // reset the counter
              UIApp.invokeCounter = 0;

              //remove the event listener
              UIApp.m_application.Idling -= UIApp.application_Idling;

              // do some dummy operation to fill the transaction;
              // empty transactions will not be displayed in undo stack 
              Line line = revit.Application.Create.NewLineBound( XYZ.Zero, XYZ.BasisX );

              DetailCurve dl = revit.ActiveUIDocument.Document.Create.NewDetailCurve( revit.ActiveUIDocument.Document.ActiveView, line );

              revit.ActiveUIDocument.Document.Delete( dl );
              revit.ActiveUIDocument.Document.Regenerate();
            }
            transaction.Commit();
          }
          catch
          {
            transaction.RollBack();
            return Result.Failed;
          }
        }
        if( UIApp.invokeCounter == 1 )  // explode your dwg in the next step
        {
          UIApp.m_application.Idling 
            += new EventHandler<IdlingEventArgs>( 
              UIApp.application_Idling );
        }
      }
      return Result.Succeeded;
    }
  }
}
