﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Windows;

using Autodesk;
using Autodesk.Revit;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Events;

using System.Diagnostics;
using System.Runtime.InteropServices;
using winform = System.Windows.Forms;

// wpf access
using System.Windows.Automation;

namespace RevitUIAutoSamples
{
  public class UIApp : IExternalApplication
  {
    [DllImport( "user32.dll", SetLastError = true, CharSet = CharSet.Auto )]
    static extern int SetWindowText( IntPtr hWnd, string lpString );

    [DllImport( "user32.dll", SetLastError = true )]
    static extern IntPtr FindWindowEx( IntPtr hwndParent, IntPtr hwndChildAfter, string lpszClass, string lpszWindow );

    [DllImport( "user32.dll", SetLastError = true )]
    public static extern IntPtr FindWindowEx( IntPtr parentHandle, IntPtr childAfter, string className, IntPtr windowTitle );

    [DllImport( "user32.dll", SetLastError = true )]
    static extern IntPtr FindWindow( string lpClassName, string lpWindowName );

    public static int invokeCounter = 0;

    private static IntPtr m_mainWndFromHandle = IntPtr.Zero;
    private static AutomationElement m_mainWndAutomationElement = null;
    private static AutomationElement m_ribbonWnd = null;

    private AutomationEventHandler m_ribbonClickHandler;
    private AutomationPropertyChangedEventHandler m_mainWindowChangeHandler;

    public static UIControlledApplication m_application = null;

    public Result OnStartup( UIControlledApplication application )
    {
      m_application = application;

      try
      {
        RibbonPanel panel = application.CreateRibbonPanel( Tab.AddIns, "UIPanel" );
        panel.Title = "UIPanel";

        PushButtonData button1 = new PushButtonData( 
          "Twice", "Import DWG Curves", 
          System.Reflection.Assembly.GetExecutingAssembly().Location, 
          typeof( Command_ImportDWGCurves ).FullName );

        panel.AddItem( button1 );

        // subscribe to Revit main window events 
        SetMainWindow();
        SubscribeMainWindowChangeEvents();

        // subscribe to Ribbon window events 
        SetRibbonWindow();
        SubscribeRibbonbarClickEvents();

      }
      catch( Exception ex )
      {
        winform.MessageBox.Show( 
          ex.StackTrace + "\r\n" + ex.InnerException, 
          "Error", winform.MessageBoxButtons.OK );

        return Result.Failed;
      }
      return Result.Succeeded;
    }

    public Result OnShutdown( UIControlledApplication application )
    {
      UnSubscribeMainWindowChangeEvents();
      UnSubscribeRibbonbarClickEvents();

      return Result.Succeeded;
    }

    public static void application_Idling( object sender, IdlingEventArgs e )
    {
      System.Threading.Thread.Sleep( 100 );

      if( invokeCounter == 1 )
      {
        Autodesk.Revit.ApplicationServices.Application app = sender as Autodesk.Revit.ApplicationServices.Application;
        UIApplication uiApp = new UIApplication( app );
        Command_ImportDWGCurves cmdDisplay = new Command_ImportDWGCurves();
        cmdDisplay.Execute( uiApp );

        PressDWGExplodeButton();
      }
    }

    public static void SetStatusText( string text )
    {
      IntPtr statusBar = FindWindowEx( m_mainWndFromHandle, IntPtr.Zero, "msctls_statusbar32", "" );

      if( statusBar != IntPtr.Zero )
      {
        SetWindowText( statusBar, text );
      }
    }

    private static void SetMainWindow()
    {
      Process[] processes = Process.GetProcessesByName( "Revit" );

      WindowHandle _hWndRevit = null;

      if( processes.Length > 0 )
      {
        m_mainWndFromHandle = processes[0].MainWindowHandle;

        _hWndRevit = new WindowHandle( m_mainWndFromHandle );
      }
      if( _hWndRevit != null )
      {
        m_mainWndAutomationElement = AutomationElement.FromHandle( _hWndRevit.Handle );
      }
    }
    public static void SetRibbonWindow()
    {
      if( m_mainWndAutomationElement != null )
      {
        PropertyCondition nameRibbonCondition = new PropertyCondition( AutomationElement.NameProperty, "RibbonHostWindow" );
        PropertyCondition typeRibbonCondition = new PropertyCondition( AutomationElement.ControlTypeProperty, ControlType.Pane );
        AndCondition andCondition = new AndCondition( typeRibbonCondition, nameRibbonCondition );

        m_ribbonWnd = m_mainWndAutomationElement.FindFirst( TreeScope.Children, andCondition );
      }
    }

    public void SubscribeMainWindowChangeEvents()
    {
      Automation.AddAutomationPropertyChangedEventHandler( 
        m_mainWndAutomationElement,
        TreeScope.Element,
        m_mainWindowChangeHandler = new AutomationPropertyChangedEventHandler( OnMainWindowResizedEvent ),
        AutomationElement.BoundingRectangleProperty/*AutomationElement.IsEnabledProperty*/ );

    }
    public void UnSubscribeMainWindowChangeEvents()
    {
      if( m_mainWindowChangeHandler != null )
      {
        Automation.RemoveAutomationPropertyChangedEventHandler( 
          m_mainWndAutomationElement, 
          m_mainWindowChangeHandler );

        m_mainWindowChangeHandler = null;
      }

    }
    private void OnMainWindowResizedEvent( 
      object src, 
      AutomationPropertyChangedEventArgs e )
    {
      AutomationElement sourceElement = src as AutomationElement;

      if( e.Property == AutomationElement.BoundingRectangleProperty )
      {
        Rect rect = ( Rect ) ( e.NewValue );

        if( rect != null )
        {
          SetStatusText( 
            "Main window has been modified, new values are W: " 
            + rect.Width + ", H: " + rect.Height );
        }
      }
    }

    public void SubscribeRibbonbarClickEvents()
    {
      if( m_ribbonWnd != null )
      {
        m_ribbonClickHandler = new AutomationEventHandler( 
          OnRibbonbarClickedEvent );

        Automation.AddAutomationEventHandler(

          // get the mouse clicks
          InvokePattern.InvokedEvent,

          // ribbonbar AutomationElement
          m_ribbonWnd,

          // ribbonbar is just a container of containers, so we need the descendants
          // (instead of TreeScope.Element or TreeScope.Children):
          TreeScope.Descendants,

          m_ribbonClickHandler
        );
      }
    }


    private void OnRibbonbarClickedEvent( object src, AutomationEventArgs e )
    {
      AutomationElement sourceElement;
      try
      {
        sourceElement = src as AutomationElement;

        if( e.EventId == InvokePattern.InvokedEvent )
        {

          string itemInfo = String.Empty;

          if( sourceElement.Current.AutomationId != null )
          {
            itemInfo += "AutoID: " + sourceElement.Current.AutomationId + ", ";
          }

          if( sourceElement.Current.ControlType.LocalizedControlType != null )
          {
            itemInfo += "Type: " + sourceElement.Current.ControlType.LocalizedControlType + ", ";
          }

          try
          {
            System.Windows.Point p = sourceElement.GetClickablePoint();
            itemInfo += "Point: (" + p.X + "/" + p.Y + ")";
          }
          catch
          {
            // no clickable point available
          }

          SetStatusText( itemInfo );
        }
        else
        {
          SetStatusText( "other than clicking event" );
        }
      }
      catch( Exception ex )
      {
        SetStatusText( ex.Message );
        return;
      }

    }
    private void UnSubscribeRibbonbarClickEvents()
    {
      if( m_ribbonWnd != null )
      {
        if( m_ribbonClickHandler != null )
        {
          Automation.RemoveAutomationEventHandler( InvokePattern.InvokedEvent,
              m_ribbonWnd, m_ribbonClickHandler );
        }
      }
    }

    public static void PressDWGExplodeButton()
    {
      PropertyCondition aidCondition = new PropertyCondition( AutomationElement.AutomationIdProperty, "2288" );

      AutomationElement explodeButton = m_mainWndAutomationElement.FindFirst( TreeScope.Children, aidCondition );
      if( explodeButton != null )
      {
        try
        {
          InvokePattern invPattern = explodeButton.GetCurrentPattern( InvokePattern.Pattern ) as InvokePattern;

          invPattern.Invoke();

          // pause
          System.Threading.Thread.Sleep( 500 );
        }
        catch( Exception ex )
        {
          SetStatusText( ex.Message );
        }
      }
    }
  }
}
